<?php

namespace Drupal\straker_translate\Element;

use Drupal\Core\Url;
use Drupal\straker_translate\StrakerTranslate;
use Drupal\config_translation\ConfigMapperInterface;
use Drupal\config_translation\ConfigEntityMapper;
use Drupal\config_translation\ConfigFieldMapper;

/**
 * Trait for straker_translate_target_status and straker_translate_target_statuses reuse.
 */
trait StrakerTranslateSourceTrait {

  /**
   *
   */
  protected function getSourceStatusForConfigMapper(ConfigMapperInterface $mapper) {
    $is_config_entity = $mapper instanceof ConfigEntityMapper;
    $entity = $is_config_entity ? $mapper->getEntity() : NULL;

    $translationService = \Drupal::service('straker_translate.config_translation');
    $source_status = $is_config_entity ?
      $translationService->getSourceStatus($entity) :
      $translationService->getConfigSourceStatus($mapper);

    return $source_status;
  }

  /**
   *
   */
  protected function getSourceStatusTextForConfigMapper(ConfigMapperInterface $mapper, $status) {
    $is_config_entity = $mapper instanceof ConfigEntityMapper;
    $entity = $is_config_entity ? $mapper->getEntity() : NULL;
    $translationService = \Drupal::service('straker_translate.config_translation');

    switch ($status) {
      case StrakerTranslate::STATUS_UNTRACKED:
      case StrakerTranslate::STATUS_REQUEST:
        return $this->t('Upload');

      case StrakerTranslate::STATUS_DISABLED:
        return $this->t('Disabled, cannot request translation');

      case StrakerTranslate::STATUS_READY:
        return $this->t('Project Created, Check Status Pending');

      case StrakerTranslate::STATUS_EDITED:
        return ($is_config_entity ? $translationService->getDocumentId($entity) : $translationService->getConfigDocumentId($mapper)) ?
          $this->t('Re-upload (content has changed since last upload)') : $this->t('Upload');

      case StrakerTranslate::STATUS_IMPORTING:
        return $this->t('Token payment pending');

      case StrakerTranslate::STATUS_PROCESSING:
        return $this->t('Project created, Translation in progress');

      case StrakerTranslate::STATUS_CURRENT:
        return $this->t('Source Translated');

      case StrakerTranslate::STATUS_ERROR:
        return $this->t('Error');

      case StrakerTranslate::STATUS_CANCELLED:
        return $this->t('Cancelled by user');

      case StrakerTranslate::STATUS_ARCHIVED:
        return $this->t('This document was archived in Straker Translate. Re-upload to translate.');

      case StrakerTranslate::STATUS_DELETED:
        return $this->t('This document was deleted in Straker Translate. Re-upload to translate.');

      default:
        return ucfirst(strtolower($status));
    }
  }

  /**
   *
   */
  protected function getSourceActionUrlForConfigMapper(ConfigMapperInterface $mapper) {
    $url = NULL;
    $is_config_entity = $mapper instanceof ConfigEntityMapper;
    $entity = $is_config_entity ? $mapper->getEntity() : NULL;

    $translationService = \Drupal::service('straker_translate.config_translation');
    $source_status = $is_config_entity ?
      $translationService->getSourceStatus($entity) :
      $translationService->getConfigSourceStatus($mapper);

    $args = $this->getActionUrlArguments($mapper);
    $document_id = $mapper instanceof ConfigEntityMapper ?
      $translationService->getDocumentId($mapper->getEntity()) :
      $translationService->getConfigDocumentId($mapper);

    if (in_array($source_status, [StrakerTranslate::STATUS_READY, StrakerTranslate::STATUS_PROCESSING])) {
      $args['document_id'] = $document_id;
      $url = Url::fromRoute('straker_translate.config.check_upload',
        $args,
        ['query' => $this->getDestinationWithQueryArray()]);
    }
    if (in_array($source_status, [StrakerTranslate::STATUS_EDITED, StrakerTranslate::STATUS_UNTRACKED, StrakerTranslate::STATUS_ERROR, StrakerTranslate::STATUS_CANCELLED, StrakerTranslate::STATUS_ARCHIVED, StrakerTranslate::STATUS_DELETED])) {
      if ($document_id) {
        $url = Url::fromRoute('straker_translate.config.update',
          $args,
          ['query' => $this->getDestinationWithQueryArray()]);
      }
      else {
        $url = Url::fromRoute('straker_translate.config.upload',
          $args,
          ['query' => $this->getDestinationWithQueryArray()]);
      }
    }
    return $url;
  }

  /**
   *
   */
  protected function getActionUrlArguments(ConfigMapperInterface &$mapper) {
    /** @var \Drupal\config_translation\ConfigEntityMapper $mapper */
    $config_mapper = $mapper;
    $args = [
      'entity_type' => $config_mapper->getPluginId(),
      'entity_id' => $config_mapper->getPluginId(),
    ];
    if ($mapper instanceof ConfigEntityMapper && !$mapper instanceof ConfigFieldMapper) {
      $args['entity_id'] = $mapper->getEntity()->id();
    }
    elseif ($mapper instanceof ConfigFieldMapper) {
      $args['entity_type'] = $mapper->getType();
      $args['entity_id'] = $mapper->getEntity()->id();
    }
    return $args;
  }

  /**
   *
   */
  protected function getSecondarySourceActionUrlsForConfigMapper(ConfigMapperInterface $mapper, $source_status, $language) {
    $actions = [];
    $translationService = \Drupal::service('straker_translate.config_translation');
    $document_id = $mapper instanceof ConfigEntityMapper ?
      $translationService->getDocumentId($mapper->getEntity()) :
      $translationService->getConfigDocumentId($mapper);
    $args = $this->getActionUrlArguments($mapper);
    if ($document_id && in_array($source_status, [StrakerTranslate::STATUS_READY, StrakerTranslate::STATUS_PROCESSING, StrakerTranslate::STATUS_ERROR])) {
      $args['document_id'] = $document_id;
      $actions[] = [
        'title' => $this->t('Check upload status'),
        'url' => Url::fromRoute('straker_translate.config.check_upload',
          $args,
          ['query' => $this->getDestinationWithQueryArray()]),
        'new_window' => FALSE,
      ];
    }
    if (in_array($source_status, [
      StrakerTranslate::STATUS_EDITED,
      StrakerTranslate::STATUS_UNTRACKED,
      StrakerTranslate::STATUS_ERROR,
      StrakerTranslate::STATUS_CANCELLED,
      StrakerTranslate::STATUS_ARCHIVED,
      StrakerTranslate::STATUS_DELETED,
    ])) {
      if ($document_id) {
        $actions[] = [
          'title' => $this->t('Update document'),
          'url' => Url::fromRoute('straker_translate.config.update',
            $args,
            ['query' => $this->getDestinationWithQueryArray()]),
          'new_window' => FALSE,
        ];
      }
      else {
        $actions[] = [
          'title' => $this->t('Upload document'),
          'url' => Url::fromRoute('straker_translate.config.upload',
            $args,
            ['query' => $this->getDestinationWithQueryArray()]),
          'new_window' => FALSE,
        ];
      }
    }
    if ($source_status == StrakerTranslate::STATUS_CURRENT || $source_status == StrakerTranslate::STATUS_EDITED) {
      $actions[] = [
        'title' => $this->t('Open in Verify'),
        'url' => Url::fromRoute('straker_translate.workbench', [
          'doc_id' => $document_id,
        ]),
        'new_window' => TRUE,
      ];
      $actions[] = [
        'title' => $this->t('Re-upload document'),
        'url' => Url::fromRoute('straker_translate.config.upload',
          $args,
          ['query' => $this->getDestinationWithQueryArray()]),
        'new_window' => FALSE,
      ];
      if ($document_id) {
        $actions[] = [
          'title' => $this->t('Re-Check document'),
          'url' => Url::fromRoute('straker_translate.config.upload',
            $args,
            ['query' => $this->getDestinationWithQueryArray()]),
          'new_window' => FALSE,
        ];
      }
    }
    return $actions;
  }

}
