<?php

namespace Drupal\straker_translate\Element;

use Drupal\config_translation\ConfigEntityMapper;
use Drupal\config_translation\ConfigFieldMapper;
use Drupal\config_translation\ConfigMapperInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Render\Element\RenderElement;
use Drupal\Core\Url;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\straker_translate\StrakerTranslate;

/**
 * Provides a Straker Translate target status element.
 *
 * @RenderElement("straker_translate_target_statuses")
 */
class StrakerTranslateTargetStatuses extends RenderElement {

  use StrakerTranslateTargetTrait;

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return [
      '#pre_render' => [
        [$this, 'preRender'],
      ],
      '#theme' => 'straker_translate_target_statuses',
      '#attached' => [
        'library' => [
          'straker_translate/straker_translate',
          'straker_translate/straker_translate.target_actions',
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

  /**
   * Calculates the url and status title and adds them to the render array.
   *
   * @param array $element
   *   The element as a render array.
   *
   * @return array
   *   The element as a render array.
   */
  public function preRender(array $element) {
    $statuses = NULL;
    if (isset($element['#entity'])) {
      $statuses = $this->getTranslationsStatuses($element['#entity'], $element['#source_langcode'], $element['#statuses']);
    }
    if (isset($element['#mapper'])) {
      $statuses = $this->getTranslationsStatusesForConfigMapper($element['#mapper'], $element['#source_langcode'], $element['#statuses']);
    }
    elseif (isset($element['#ui_component'])) {
      $statuses = $this->getTranslationsStatusesForUI($element['#ui_component'], $element['#source_langcode'], $element['#statuses']);
    }
    $element['#statuses'] = $statuses;
    return $element;
  }

  /**
   * Gets the translation status of an entity in a format ready to display.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param string $source_langcode
   *   The source language code.
   * @param array $statuses
   *   Array of known statuses keyed by language code.
   *
   * @return array
   *   A render array.
   */
  protected function getTranslationsStatuses(ContentEntityInterface &$entity, $source_langcode, array $statuses) {
    $translations = [];
    $languages = \Drupal::languageManager()->getLanguages();
    $languages = array_filter($languages, function (LanguageInterface $language) {
      $configLanguage = ConfigurableLanguage::load($language->getId());
      return \Drupal::service('straker_translate.configuration')->isLanguageEnabled($configLanguage);
    });
    foreach ($statuses as $langcode => $status) {
      if ($langcode !== $source_langcode && array_key_exists($langcode, $languages)) {
        // We may have an existing translation already.
        if ($entity->hasTranslation($langcode) && $status == StrakerTranslate::STATUS_REQUEST) {
          $translations[$langcode] = [
            'status' => StrakerTranslate::STATUS_UNTRACKED,
            'url' => $this->getTargetActionUrl($entity, StrakerTranslate::STATUS_UNTRACKED, $langcode),
            'actions' => $this->getSecondaryTargetActionUrls($entity, StrakerTranslate::STATUS_UNTRACKED, $langcode),
            'new_window' => FALSE,
          ];
        }
        else {
          $translations[$langcode] = [
            'status' => $status,
            'url' => $this->getTargetActionUrl($entity, $status, $langcode),
            'actions' => $this->getSecondaryTargetActionUrls($entity, $status, $langcode),
            'new_window' => in_array($status, [StrakerTranslate::STATUS_CURRENT, StrakerTranslate::STATUS_INTERMEDIATE, StrakerTranslate::STATUS_EDITED]),
          ];
        }
      }
      array_walk($languages, function ($language, $langcode) use ($entity, &$translations) {
        if (!isset($translations[$langcode]) &&
            $langcode !== $entity->getUntranslated()->language()->getId()) {
          $translations[$langcode] = [
            'status' => StrakerTranslate::STATUS_REQUEST,
            'url' => $this->getTargetActionUrl($entity, StrakerTranslate::STATUS_REQUEST, $langcode),
            'actions' => $this->getSecondaryTargetActionUrls($entity, StrakerTranslate::STATUS_REQUEST, $langcode),
            'new_window' => FALSE,
          ];
        }
      });
    }
    foreach ($languages as $langcode => $language) {
      // Show the untracked translations in the bulk management form, unless it's the
      // source one.
      if (!isset($translations[$langcode]) && $entity->hasTranslation($langcode) && $source_langcode !== $langcode) {
        $translations[$langcode] = [
          'status' => StrakerTranslate::STATUS_UNTRACKED,
          'url' => NULL,
          'actions' => $this->getSecondaryTargetActionUrls($entity, StrakerTranslate::STATUS_UNTRACKED, $langcode),
          'new_window' => FALSE,
        ];
      }
    }
    ksort($translations);
    foreach ($translations as $langcode => &$translation) {
      $translation['status_text'] = $this->getTargetStatusText($translation['status'], $langcode, $entity->hasTranslation($langcode));
      $translation['language'] = $langcode;
    }
    return $translations;
  }

  /**
   *
   */
  protected function getTranslationsStatusesForConfigMapper(ConfigMapperInterface &$mapper, $source_langcode, array $statuses) {
    $translations = [];
    foreach ($statuses as $langcode => &$status) {
      $status['actions'] = $this->getSecondaryTargetActionUrlsForConfigMapper($mapper, $status['status'], $status['language']);
    }
    return $statuses;
    $languages = \Drupal::languageManager()->getLanguages();
    $languages = array_filter($languages, function (LanguageInterface $language) {
      $configLanguage = ConfigurableLanguage::load($language->getId());
      return \Drupal::service('straker_translate.configuration')->isLanguageEnabled($configLanguage);
    });
    foreach ($statuses as $langcode => $status) {
      if ($langcode !== $source_langcode && array_key_exists($langcode, $languages)) {
        // We may have an existing translation already.
        if ($mapper instanceof ConfigEntityMapper && $mapper->getEntity()->hasTranslation($langcode) && $status == StrakerTranslate::STATUS_REQUEST) {
          $translations[$langcode] = [
            'status' => StrakerTranslate::STATUS_UNTRACKED,
            'url' => $this->getTargetActionUrlForConfigMapper($mapper, StrakerTranslate::STATUS_UNTRACKED, $langcode),
            'actions' => $this->getSecondaryTargetActionUrlsForConfigMapper($mapper, StrakerTranslate::STATUS_UNTRACKED, $langcode),
            'new_window' => FALSE,
          ];
        }
        else {
          $translations[$langcode] = [
            'status' => $status,
            'url' => $this->getTargetActionUrlForConfigMapper($mapper, $status, $langcode),
            'actions' => $this->getSecondaryTargetActionUrlsForConfigMapper($mapper, $status, $langcode),
            'new_window' => in_array($status, [StrakerTranslate::STATUS_CURRENT, StrakerTranslate::STATUS_INTERMEDIATE, StrakerTranslate::STATUS_EDITED]),
          ];
        }
      }
      array_walk($languages, function ($language, $langcode) use ($mapper, &$translations) {
        if ($mapper instanceof ConfigEntityMapper) {
          if (!isset($translations[$langcode]) &&
            $langcode !== $mapper->getEntity()
              ->getUntranslated()
              ->language()
              ->getId()) {
            $translations[$langcode] = [
              'status' => StrakerTranslate::STATUS_REQUEST,
              'url' => $this->getTargetActionUrlForConfigMapper($mapper, StrakerTranslate::STATUS_REQUEST, $langcode),
              'actions' => $this->getSecondaryTargetActionUrlsForConfigMapper($mapper, StrakerTranslate::STATUS_REQUEST, $langcode),
              'new_window' => FALSE,
            ];
          }
        }
      });
    }
    foreach ($languages as $langcode => $language) {
      // Show the untracked translations in the bulk management form, unless it's the
      // source one.
      if (!isset($translations[$langcode]) && $mapper instanceof ConfigEntityMapper && $mapper->getEntity()->hasTranslation($langcode) && $source_langcode !== $langcode) {
        $translations[$langcode] = [
          'status' => StrakerTranslate::STATUS_UNTRACKED,
          'url' => NULL,
          'actions' => $this->getSecondaryTargetActionUrlsForConfigMapper($mapper, StrakerTranslate::STATUS_UNTRACKED, $langcode),
          'new_window' => FALSE,
        ];
      }
    }
    ksort($translations);
    foreach ($translations as $langcode => &$translation) {
      $hasTranslation = $mapper instanceof ConfigEntityMapper ? $mapper->getEntity()->hasTranslation($langcode) : FALSE;
      $translation['status_text'] = $this->getTargetStatusText($translation['status'], $langcode, $hasTranslation);
      $translation['language'] = $langcode;
    }
    return $translations;
  }

  /**
   *
   */
  protected function getTranslationsStatusesForUI($component, $source_langcode, array $statuses) {
    $translations = [];
    $languages = \Drupal::languageManager()->getLanguages();
    $languages = array_filter($languages, function (LanguageInterface $language) {
      $configLanguage = ConfigurableLanguage::load($language->getId());
      return \Drupal::service('straker_translate.configuration')->isLanguageEnabled($configLanguage);
    });
    foreach ($statuses as $langcode => $status) {
      if ($langcode !== $source_langcode && array_key_exists($langcode, $languages)) {
        // We may have an existing translation already.
        $translations[$langcode] = [
          'status' => $status,
          'url' => $this->getTargetActionUrlForUI($component, $status, $langcode),
          'new_window' => in_array($status, [StrakerTranslate::STATUS_CURRENT, StrakerTranslate::STATUS_INTERMEDIATE, StrakerTranslate::STATUS_EDITED]),
        ];
      }
      array_walk($languages, function ($language, $langcode) use ($component, &$translations) {
        if (!isset($translations[$langcode]) &&
          $langcode !== 'en') {
          $translations[$langcode] = [
            'status' => StrakerTranslate::STATUS_REQUEST,
            'url' => $this->getTargetActionUrlForUI($component, StrakerTranslate::STATUS_REQUEST, $langcode),
            'new_window' => FALSE,
          ];
        }
      });
    }
    ksort($translations);
    foreach ($translations as $langcode => &$translation) {
      $translation['status_text'] = $this->getTargetStatusText($translation['status'], $langcode);
      $translation['language'] = $langcode;
    }
    return $translations;
  }

  /**
   *
   */
  protected function getTargetActionUrlForUI($component, $target_status, $langcode) {
    $url = NULL;
    $document_id = \Drupal::service('straker_translate.interface_translation')
      ->getDocumentId($component);
    if ($document_id) {
      if ($target_status == StrakerTranslate::STATUS_READY || $target_status == StrakerTranslate::STATUS_ERROR) {
        $url = Url::fromRoute('straker_translate.interface_translation.download', [],
          [
            'query' => [
              'component' => $component,
              'locale' => $langcode,
            ] + $this->getDestinationWithQueryArray(),
          ]);
      }
      if ($target_status == StrakerTranslate::STATUS_CURRENT ||
        $target_status == StrakerTranslate::STATUS_INTERMEDIATE ||
        $target_status == StrakerTranslate::STATUS_EDITED) {
        $url = Url::fromRoute('straker_translate.workbench', [
          'doc_id' => $document_id,
        ]);
      }
      if ($target_status == StrakerTranslate::STATUS_DISABLED) {
        $url = NULL;
      }
    }
    return $url;
  }

  /**
   * Get secondary target actions, which will be shown when expanded.
   *
   * @param \Drupal\config_translation\ConfigMapperInterface $mapper
   *   The entity.
   * @param string $target_status
   *   The target status.
   * @param string $langcode
   *   The language code.
   *
   * @return array
   *   Array of links.
   */
  protected function getSecondaryTargetActionUrlsForConfigMapper(ConfigMapperInterface &$mapper, $target_status, $langcode) {
    $url = NULL;
    $target_status = strtoupper($target_status);
    $language = \Drupal::languageManager()->getLanguage($langcode);
    $translationService = \Drupal::service('straker_translate.config_translation');
    /** @var \Drupal\Core\Config\Entity\ConfigEntityInterface $entity */
    $entity = $mapper instanceof ConfigEntityMapper ? $mapper->getEntity() : NULL;
    $document_id = $mapper instanceof ConfigEntityMapper ?
      $translationService->getDocumentId($entity) :
      $translationService->getConfigDocumentId($mapper);
    $args = $this->getActionUrlArgumentsForConfigMapper($mapper);

    $actions = [];
    if ($document_id) {
      if ($target_status == StrakerTranslate::STATUS_READY || $target_status == StrakerTranslate::STATUS_ERROR) {
        $file_id = $mapper instanceof ConfigEntityMapper ?
            $translationService->getTargetFileId($mapper->getEntity(), $langcode) :
            $translationService->getConfigTargetFileId($mapper, $langcode);
        $args['file_id'] = $file_id;
        $args['langcode'] = $langcode;
        if ($target_status == StrakerTranslate::STATUS_READY) {
          $actions[] = [
            'title' => $this->t('Download translation'),
            'url' => Url::fromRoute('straker_translate.config.download',
              $args,
              ['query' => $this->getDestinationWithQueryArray()]),
            'new_window' => FALSE,
          ];
        }
        if ($file_id && $target_status == StrakerTranslate::STATUS_ERROR) {
          $actions[] = [
            'title' => $this->t('Download again translation'),
            'url' => Url::fromRoute('straker_translate.config.download',
              $args,
              ['query' => $this->getDestinationWithQueryArray()]),
            'new_window' => FALSE,
          ];
        }
        // @todo add url for disassociate.
      }

      if (in_array($target_status, [
        StrakerTranslate::STATUS_CURRENT,
        StrakerTranslate::STATUS_INTERMEDIATE,
        StrakerTranslate::STATUS_EDITED,
        StrakerTranslate::STATUS_READY,
      ], TRUE)) {
        $actions[] = [
          'title' => $this->t('Open in Verify'),
          'url' => Url::fromRoute('straker_translate.workbench', [
            'doc_id' => $document_id,
          ]),
          'new_window' => TRUE,
        ];
      }
    }

    return $actions;
  }

  /**
   *
   */
  protected function getActionUrlArgumentsForConfigMapper(ConfigMapperInterface &$mapper) {
    /** @var \Drupal\config_translation\ConfigEntityMapper $mapper */
    $args = [
      'entity_type' => $mapper->getPluginId(),
      'entity_id' => $mapper->getPluginId(),
    ];
    if ($mapper instanceof ConfigEntityMapper && !$mapper instanceof ConfigFieldMapper) {
      $args['entity_id'] = $mapper->getEntity()->id();
    }
    elseif ($mapper instanceof ConfigFieldMapper) {
      $args['entity_type'] = $mapper->getType();
      $args['entity_id'] = $mapper->getEntity()->id();
    }
    return $args;
  }

}
