<?php

namespace Drupal\straker_translate\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\straker_translate\StrakerTranslateProfileInterface;

/**
 * Defines the Straker Translate Profile entity.
 *
 * @ConfigEntityType(
 *   id = "straker_translate_profile",
 *   label = @Translation("Straker Translate Profile"),
 *   handlers = {
 *     "list_builder" = "Drupal\straker_translate\StrakerTranslateProfileListBuilder",
 *     "form" = {
 *       "add" = "Drupal\straker_translate\Form\StrakerTranslateProfileAddForm",
 *       "edit" = "Drupal\straker_translate\Form\StrakerTranslateProfileEditForm",
 *       "delete" = "Drupal\straker_translate\Form\StrakerTranslateProfileDeleteForm"
 *     },
 *   },
 *   admin_permission = "administer straker_translate",
 *   config_prefix = "profile",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "weight" = "weight"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "weight",
 *     "locked",
 *     "auto_upload",
 *     "auto_download",
 *     "auto_download_worker",
 *     "append_type_to_title",
 *     "workflow",
 *     "token_confirmation",
 *     "language_overrides",
 *   },
 *   links = {
 *     "add-form" = "/admin/straker-translate/settings/profile/add",
 *     "delete-form" = "/admin/straker-translate/settings/profile/{profile}/delete",
 *     "edit-form" = "/admin/straker-translate/settings/profile/{profile}/edit",
 *   },
 * )
 */
class StrakerTranslateProfile extends ConfigEntityBase implements StrakerTranslateProfileInterface {

  /**
   * The profile ID (machine name).
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable label for the profile.
   *
   * @var string
   */
  protected $label;

  /**
   * The weight of the profile, used in lists of profiles.
   *
   * @var int
   */
  protected $weight = 0;

  /**
   * Locked profiles cannot be edited.
   *
   * @var bool
   */
  protected $locked = FALSE;

  /**
   * Entities using this profile may automatically upload sources.
   *
   * @var bool
   */
  protected $auto_upload = FALSE;

  /**
   * Entities using this profile may automatically download translations.
   *
   * @var bool
   */
  protected $auto_download = FALSE;

  /**
   * Entities using this profile may use a worker queue to download translations.
   *
   * @var bool
   */
  protected $auto_download_worker = FALSE;

  /**
   * Entities using this profile will use this project.
   *
   * @var string
   */
  protected $project = 'default';

  /**
   * Entities using this profile will use this workflow.
   *
   * @var string
   */
  protected $workflow = 'default';

  /**
   * If confirmation is required before uploading to Translate.
   *
   * @var string
   */
  protected $token_confirmation = 'false';

  /**
   * Specific target language settings override.
   *
   * @var array
   */
  protected $language_overrides = [];

  /**
   * If content type is to be appended to title when uploading to Translate.
   *
   * @var string
   */
  protected $append_type_to_title = 'global_setting';

  /**
   * {@inheritdoc}
   */
  public function __construct(array $values, $entity_type) {
    parent::__construct($values, $entity_type);
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    // getWorkflow() could return 'default', but we need to check if the default itself is 'project_default' as well.
    $default_workflow = \Drupal::config('straker_translate.account')->get('default.workflow');
    if ($this->getWorkflow() === 'project_default' || $default_workflow === 'project_default') {
      foreach ($this->language_overrides as $langcode => $v) {
        if (isset($this->language_overrides[$langcode]['custom']['workflow'])) {
          unset($this->language_overrides[$langcode]['custom']['workflow']);
        }
      }
    }
    parent::preSave($storage);
  }

  /**
   * {@inheritdoc}
   */
  public function isLocked() {
    return (bool) $this->locked;
  }

  /**
   * {@inheritdoc}
   */
  public function getAppendContentTypeToTitle() {
    return $this->append_type_to_title;
  }

  /**
   * {@inheritdoc}
   */
  public function setAppendContentTypeToTitle($append_type_to_title = 'global_setting') {
    $this->append_type_to_title = $append_type_to_title;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight() {
    return $this->weight;
  }

  /**
   * {@inheritdoc}
   */
  public function setWeight($weight) {
    $this->weight = $weight;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAutomaticUpload() {
    return (bool) $this->auto_upload;
  }

  /**
   * {@inheritdoc}
   */
  public function setAutomaticUpload($auto_upload) {
    $this->auto_upload = $auto_upload;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAutomaticDownloadWorker() {
    return (bool) $this->auto_download_worker;
  }

  /**
   * {@inheritdoc}
   */
  public function setAutomaticDownloadWorker($auto_download_worker) {
    $this->auto_download_worker = $auto_download_worker;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAutomaticDownload() {
    return (bool) $this->auto_download;
  }

  /**
   * {@inheritdoc}
   */
  public function setAutomaticDownload($auto_download) {
    $this->auto_download = $auto_download;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWorkflow() {
    return $this->workflow;
  }

  /**
   * {@inheritdoc}
   */
  public function setWorkflow($workflow) {
    $this->workflow = $workflow;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAutomaticDownloadForTarget($langcode) {
    $auto_download = $this->hasAutomaticDownload();
    if (isset($this->language_overrides[$langcode]) && $this->hasCustomSettingsForTarget($langcode)) {
      $auto_download = $this->language_overrides[$langcode]['custom']['auto_download'];
    }
    if ($this->hasDisabledTarget($langcode)) {
      $auto_download = FALSE;
    }
    return $auto_download;
  }

  /**
   * {@inheritdoc}
   */
  public function setAutomaticDownloadForTarget($langcode, $value) {
    $this->language_overrides[$langcode]['custom']['auto_download'] = $value;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasCustomSettingsForTarget($langcode) {
    return isset($this->language_overrides[$langcode]) && $this->language_overrides[$langcode]['overrides'] === 'custom';
  }

  /**
   * {@inheritdoc}
   */
  public function getCustomSettingsForTarget($langcode) {
    return $this->language_overrides[$langcode]['overrides'] ?? 'default';
  }

  /**
   * {@inheritdoc}
   */
  public function hasDisabledTarget($langcode) {
    return isset($this->language_overrides[$langcode]) && $this->language_overrides[$langcode]['overrides'] === 'disabled';
  }

}
