<?php

namespace Drupal\straker_translate\FormComponent;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Straker Translate form-component plugins.
 *
 * @package Drupal\straker_translate\FormComponent
 */
abstract class StrakerTranslateFormComponentBase extends PluginBase implements StrakerTranslateFormComponentInterface {

  use DependencySerializationTrait;

  /**
   * The entity_type.manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language_manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The straker_translate.configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translateConfiguration;

  /**
   * The straker_translate.content_translation service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface
   */
  protected $translationService;

  /**
   * The entity-type ID.
   *
   * @var bool|string|null
   */
  protected $entityTypeId = FALSE;

  /**
   * An array of entity-type definitions.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface[]
   */
  protected $entityTypes = [];

  /**
   * StrakerTranslateFormComponentBase constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity_type.manager service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language_manager service.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The straker_translate.configuration service.
   * @param \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $translation_service
   *   The straker_translate.content_translation service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    StrakerTranslateConfigurationServiceInterface $straker_translate_configuration,
    StrakerTranslateContentTranslationServiceInterface $translation_service
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->straker_translateConfiguration = $straker_translate_configuration;
    $this->translationService = $translation_service;
  }

  /**
   * Prevents serializing the container in children when the form is submitted.
   *
   * For some reason DependencySerializationTrait doesn't do the trick. Because
   * these classes should get serialized only when performing a batch operation,
   * and because they're not needed when the form object is unserialized, there
   * should be no need to implement __wakeup().
   *
   * @return array
   *   The properties to maintain.
   */
  public function __sleep(): array {
    return parent::__sleep();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.content_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(array $arguments = []) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityTypeId(?string $entity_type_id) {
    if (!$this->entityTypeId) {
      $this->entityTypeId = $entity_type_id;
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityType(string $entity_type_id) {
    if (!($this->entityTypes[$entity_type_id] ?? NULL)) {
      $this->entityTypes[$entity_type_id] = $this->entityTypeManager->getDefinition($entity_type_id);
    }

    return $this->entityTypes[$entity_type_id];
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle() {
    return $this->pluginDefinition['title'];
  }

  /**
   * {@inheritdoc}
   */
  public function getGroup() {
    if ($group = $this->pluginDefinition['group'] ?? NULL) {
      return $group;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getGroupMachineName() {
    if ($group = $this->pluginDefinition['group'] ?? NULL) {
      if ($group instanceof TranslatableMarkup) {
        $group = $group->getUntranslatedString();
      }

      return preg_replace("/[^a-z0-9_]+/", '_', strtolower($group));
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight() {
    return $this->pluginDefinition['weight'] ?? 0;
  }

}
