<?php

namespace Drupal\straker_translate\Plugin\QueueWorker;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\straker_translate\StrakerTranslate;
use Drupal\Component\Render\FormattableMarkup;

/**
 * Provides base functionality for the Straker Translate Queue Workers.
 *
 * @QueueWorker(
 *   id = "straker_translate_downloader_queue_worker",
 *   title = @Translation("Straker Translate Download Queue"),
 *   cron = {"time" = 60}
 * )
 */
class StrakerTranslateDownloaderQueueWorker extends QueueWorkerBase {

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $langcode = $data['langcode'];
    $entity_type_id = $data['entity_type_id'];
    $entity_id = $data['entity_id'];
    $document_id = $data['document_id'];
    $file_id = $data['file_id'];

    $entity = \Drupal::entityTypeManager()
      ->getStorage($entity_type_id)
      ->load($entity_id);
    if ($entity instanceof ConfigEntityInterface) {
      /** @var \Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface $translation_service */
      $translation_service = \Drupal::service('straker_translate.config_translation');
    }
    elseif ($entity instanceof ContentEntityInterface) {
      /** @var \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $translation_service */
      $translation_service = \Drupal::service('straker_translate.content_translation');
    }
    if (empty($translation_service)) {
      \Drupal::logger('straker_translate')
        ->error('Can not download - entity (@instance) is not supported instance of a class', [
          '@instance' => gettype($entity),
        ]
        );
      throw new \Exception(new FormattableMarkup('Can not download - entity (@instance) is not supported instance of a class', [
        '@instance' => gettype($entity),
      ])
      );
    }
    $target_status = $translation_service->getTargetStatus($entity, $langcode);
    if ($target_status == StrakerTranslate::STATUS_CURRENT) {
      \Drupal::logger('straker_translate')
        ->notice('Already Downloaded - target @langcode in @entity @bundle @id', [
          '@langcode' => $langcode,
          '@document' => $document_id,
          '@entity' => $entity->label(),
          '@id' => $entity->id(),
          '@bundle' => $entity->bundle(),
        ]);
      return;
    }
    $download = $translation_service->downloadDocument($entity, $file_id, $langcode);

    if ($download === FALSE || $download === NULL) {
      \Drupal::logger('straker_translate')
        ->error('No download for target @langcode happened in document @document on @entity @bundle @id.', [
          '@langcode' => (string) ($langcode ?? 'N/A'),
          '@document' => (string) ($document_id ?? 'N/A'),
          '@entity' => (string) ($entity->label() ?? 'N/A'),
          '@id'      => (string) ($entity->id() ?? 'N/A'),
          '@bundle'  => (string) ($entity->bundle() ?? 'N/A'),
        ]);
      throw new \Exception(new FormattableMarkup('No download for target @langcode happened in document @document on @entity @bundle @id.', [
        '@langcode' => $langcode,
        '@document' => $document_id,
        '@entity' => $entity->label(),
        '@id' => $entity->id(),
        '@bundle' => $entity->bundle(),
      ]));
    }
  }

}
