<?php

namespace Drupal\straker_translate\Plugin\StrakerTranslateFormComponent\BulkAction;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentBulkActionBase;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentBulkActionExecutor;
use Drupal\straker_translate\LanguageLocaleMapperInterface;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a Straker Translate bulk action plugin for the delete all translations
 * operation.
 *
 * @StrakerTranslateFormComponentBulkAction(
 *   id = "delete_translations",
 *   title = @Translation("Delete all translations"),
 *   group = @Translation("Delete translations"),
 *   weight = 90,
 *   form_ids = {
 *     "straker_translate_management",
 *     "straker_translate_entity_management",
 *   },
 *   redirect = "entity:delete-multiple-form"
 * )
 */
class DeleteTranslations extends StrakerTranslateFormComponentBulkActionBase {

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The language-locale mapper.
   *
   * @var \Drupal\straker_translate\LanguageLocaleMapperInterface
   */
  protected $languageLocaleMapper;

  /**
   * The user currently logged in.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The user's temporary store.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * AssignJobId constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity_type.manager service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language_manager service.
   * @param \Drupal\straker_translate\LanguageLocaleMapperInterface $language_locale_mapper
   *   The language-locale mapper.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The straker_translate.configuration service.
   * @param \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $translation_service
   *   The straker_translate.content_translation service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity_type.bundle.info service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The factory for the temp store object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, LanguageLocaleMapperInterface $language_locale_mapper, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, StrakerTranslateContentTranslationServiceInterface $translation_service, EntityTypeBundleInfoInterface $entity_type_bundle_info, PrivateTempStoreFactory $temp_store_factory, AccountInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $language_manager, $straker_translate_configuration, $translation_service);
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->languageLocaleMapper = $language_locale_mapper;
    $this->tempStoreFactory = $temp_store_factory;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('straker_translate.language_locale_mapper'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.content_translation'),
      $container->get('entity_type.bundle.info'),
      $container->get('tempstore.private'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(array $arguments = []) {
    $entity_type_id = $arguments['entity_type_id'] ?? NULL;
    if ($entity_type_id != NULL) {
      return $this->entityTypeManager->getDefinition($entity_type_id)
        ->hasLinkTemplate('delete-multiple-form');
    }
    return FALSE;
  }

  /**
   *
   */
  public function execute(array $entities, array $options, StrakerTranslateFormComponentBulkActionExecutor $executor) {
    $entityInfo = [];
    $languages = $this->languageManager->getLanguages();
    foreach ($entities as $entity) {
      $source_language = $entity->getUntranslated()->language();
      foreach ($languages as $langcode => $language) {
        if ($source_language->getId() !== $langcode && $entity->hasTranslation($langcode)) {
          /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
          $entityInfo[$entity->id()][$langcode] = $langcode;
        }
      }
    }
    if (!empty($entityInfo)) {
      $this->tempStoreFactory->get('entity_delete_multiple_confirm')
        ->set($this->currentUser->id() . ':' . $this->entityTypeId, $entityInfo);
    }
    else {
      $this->messenger()->addWarning($this->t('No valid translations for deletion.'));
      // Ensure selection is persisted.
      return FALSE;
    }
    return TRUE;
  }

}
