<?php

namespace Drupal\straker_translate\Remote;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;

/**
 *
 */
interface StrakerTranslateApiInterface extends ContainerInjectionInterface {

  /**
   * Get the account information.
   *
   * @return array|bool
   *   Array with account info. FALSE in case of error.
   *   The keys are user_id, username, login_id, email, and name.
   */
  public function getAccountInfo();

  /**
   * Get the available locales on Straker Translate.
   *
   * @return array|bool
   *   Array of locales (as in de-DE, es-ES). FALSE if there is an error.
   */
  public function getLocales();

  /**
   * Adds a document to Straker Translate.
   *
   * @param array $args
   *   The document data.
   *
   * @see http://devzone.straker_translate.com
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateApiException
   */
  public function addDocument($args);

  /**
   * Cofirm a document to Straker Translate.
   *
   * @param array $args
   *   The document data.
   *
   * @see http://devzone.straker_translate.com
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateApiException
   */
  public function confirmDocument($args);

  /**
   * Cancels a document on Straker Translate.
   *
   * @param string $document_id
   *   The document id.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function cancelDocument($document_id);

  /**
   * Cancels the document target with this document id and locale from the Straker Translate service.
   *
   * @param string $document_id
   *   The document id.
   * @param string $locale
   *   The locale target we want to cancel the translation.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function cancelDocumentTarget($document_id, $locale);

  /**
   * Gets the document target translation statuses from the Straker Translate service.
   *
   * @param string $id
   *   The document id.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function getDocumentTranslationStatuses($id);

  /**
   * Gets the document target translation status for a given locale from the Straker Translate service.
   *
   * @param string $id
   *   The document id.
   * @param string $locale
   *   The target locale.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function getDocumentTranslationStatus($id, $locale);

  /**
   * Gets the document status from the Straker Translate service.
   *
   * @param string $id
   *   The document id.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function getDocumentStatus($id);

  /**
   * Gets a document target translation for a given locale from the Straker Translate service.
   *
   * @param string $id
   *   The document id.
   * @param string $locale
   *   The target locale.
   * @param bool $useSource
   *   (Optional) Flag indicating if should return the source if this is not yet
   *   translated. By default is FALSE.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   A response.
   */
  public function getTranslation($id, $locale);

  /**
   * Gets the community related workflows.
   *
   * @return array|bool
   *   Array with communities information. FALSE in case of error.
   *   The keys are the ids, and values are the name of the resource.
   */
  public function getWorkflows();

}
