<?php

namespace Drupal\straker_translate;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;

/**
 *
 */
interface StrakerTranslateInterface extends ContainerInjectionInterface {

  /**
   * Get the available locales on Straker Translate with extra information.
   *
   * @return array
   *   Array of locales. Empty array if there is an error.
   *   The array has the locale as key, and the value is a nested array with
   *   the following keys: code, language_code, title, language, country_code,
   *   and country.
   */
  public function getLocalesInfo();

  /**
   * Get the account information.
   *
   * @return array|bool
   *   Array with account info. FALSE in case of error.
   *   The keys are user_id, username, login_id, email, and name.
   */
  public function getAccountInfo();

  /**
   * Gets the account related resources.
   *
   * @return array|bool
   *   Array with resources info. FALSE in case of error.
   *   The workflow is a nested array with key the id, and value the name of the resource.
   */
  public function getResources();

  /**
   * Gets the account related workflows.
   *
   * @return array|bool
   *   Array with workflows information. FALSE in case of error.
   *   The keys are the ids, and values are the name of the resource.
   */
  public function getWorkflows();

  /**
   *
   */
  public function getDefaults();

  /**
   * Uploads a document to the Straker Translate service.
   *
   * @param string $title
   *   The title of the document as it will be seen in the Verify.
   * @param string|array $content
   *   The content of the document. It can be a json string or an array that will
   *   be json encoded.
   * @param string $locale
   *   The Straker Translate locale.
   * @param string $filename
   *   (optional) The document filename in the site if any. This allows support for in-context review.
   * @param \Drupal\straker_translate\StrakerTranslateProfileInterface $profile
   *   (optional) The profile being used.
   *
   * @return string
   *   The document ID assigned to the uploaded document.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateProcessedWordsLimitException
   * @throws \Drupal\straker_translate\Exception\StrakerTranslatePaymentRequiredException
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateApiException
   */
  public function uploadDocument($title, $content, $locale, $token_confirmation, $filename, StrakerTranslateProfileInterface $profile = NULL);

  /**
   * Confirms the document with this document id from the Straker Translate service.
   *
   * @param string $project_id
   *   The project id in Straker Translate.
   *
   * @return bool
   *   TRUE if the document was successfully cancelled. FALSE if not.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentAlreadyCompletedException
   *   If the document was already completed.
   */
  public function confirmDocument($project_id);

  /**
   * Gets a document status.
   *
   * @param string $doc_id
   *   The document id.
   *
   * @return bool
   *   TRUE if the document exists. FALSE if not.
   */
  public function getDocumentStatus($doc_id);

  /**
   * Gets the last edited timestamp from Straker Translate service.
   *
   * @param string $doc_id
   *   The document id in Straker Translate.
   *
   * @return int
   *   The timestamp.
   */
  public function getUploadedTimestamp($doc_id);

  /**
   * Gets the status of the translation.
   *
   * @param string $doc_id
   *   The document ID in Straker Translate.
   * @param $locale
   *   The locale we want to know the translation status.
   *
   * @return bool|int
   *   Returns TRUE if the document translation is completed. FALSE if it was not
   *   requested. The percentage if it's still in progress.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException
   */
  public function getDocumentTranslationStatus($doc_id, $locale);

  /**
   * Gets the status of the translations.
   *
   * @param string $doc_id
   *   The document ID in Straker Translate.
   *
   * @return array
   *   Returns array keyed by the locale with the percentage of completion.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException
   */
  public function getDocumentTranslationStatuses($doc_id);

  /**
   * Gets the translation of a document for a given locale.
   *
   * @param string $doc_id
   *   The document ID in Straker Translate.
   * @param $locale
   *   The locale we want to know the translation status.
   *
   * @return array
   *   Returns array with the content of the document.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException
   */
  public function downloadDocument($doc_id, $locale);

  /**
   * Cancels the document with this document id from the Straker Translate service.
   *
   * @param string $doc_id
   *   The document id in Straker Translate.
   *
   * @return bool
   *   TRUE if the document was successfully cancelled. FALSE if not.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentAlreadyCompletedException
   *   If the document was already completed.
   */
  public function cancelDocument($doc_id);

  /**
   * Cancels the document target with this document id and locale from the Straker Translate service.
   *
   * @param string $doc_id
   *   The document id in Straker Translate.
   * @param string $locale
   *   The locale target we want to cancel the translation.
   *
   * @return bool
   *   TRUE if the document target was successfully cancelled. FALSE if not.
   *
   * @throws \Drupal\straker_translate\Exception\StrakerTranslateDocumentTargetAlreadyCompletedException
   *   If the target was already completed.
   */
  public function cancelDocumentTarget($doc_id, $locale);

}
