<?php

namespace Drupal\straker_translate;

use Drupal\config_translation\ConfigMapperManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for checking Straker Translate profiles usage.
 */
class StrakerTranslateProfileUsage implements StrakerTranslateProfileUsageInterface {

  /**
   * The Straker Translate configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translateConfiguration;

  /**
   * The configuration mapper manager.
   *
   * @var \Drupal\config_translation\ConfigMapperManagerInterface
   */
  protected $configMapperManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The Entity Type Manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new StrakerTranslateProfileUsage object.
   *
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The Straker Translate configuration service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\config_translation\ConfigMapperManagerInterface $config_mapper_manager
   *   The configuration mapper manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info.
   */
  public function __construct(StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, EntityTypeManagerInterface $entity_type_manager, ConfigMapperManagerInterface $config_mapper_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info) {
    $this->straker_translateConfiguration = $straker_translate_configuration;
    $this->entityTypeManager = $entity_type_manager;
    $this->configMapperManager = $config_mapper_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsedByContent(StrakerTranslateProfileInterface $profile) {
    $entity_query = $this->entityTypeManager->getStorage('straker_content_metadata')->getQuery();
    $entity_query->accessCheck(FALSE);
    $entity_query->condition('profile', $profile->id());
    $result = $entity_query->count()->execute();
    $used = ($result > 0) ? StrakerTranslateProfileUsageInterface::USED_BY_CONTENT : StrakerTranslateProfileUsageInterface::UNUSED;
    return $used;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsedByConfig(StrakerTranslateProfileInterface $profile) {
    $mappers = $this->configMapperManager->getMappers();

    $used = StrakerTranslateProfileUsageInterface::UNUSED;
    foreach ($mappers as $plugin_id => $mapper) {
      $config_profile = $this->straker_translateConfiguration->getConfigProfile($plugin_id, FALSE);
      if ($config_profile !== NULL && $config_profile->id() === $profile->id()) {
        $used |= StrakerTranslateProfileUsageInterface::USED_BY_CONFIG;
      }
    }

    if ($used !== StrakerTranslateProfileUsageInterface::USED_BY_CONFIG) {
      $entity_query = $this->entityTypeManager->getStorage('straker_config_metadata')->getQuery();
      $entity_query->accessCheck(FALSE);
      $entity_query->condition('profile', $profile->id());
      $result = $entity_query->count()->execute();
      $used = ($result > 0) ? StrakerTranslateProfileUsageInterface::USED_BY_CONFIG : StrakerTranslateProfileUsageInterface::UNUSED;
    }

    return $used;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsedByContentSettings(StrakerTranslateProfileInterface $profile) {
    $entity_types = $this->straker_translateConfiguration->getEnabledEntityTypes();

    $used = StrakerTranslateProfileUsageInterface::UNUSED;
    foreach ($entity_types as $entity_type_id => $entity_type_definition) {
      $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      foreach ($bundles as $bundle_id => $bundle_definition) {
        $config_profile = $this->straker_translateConfiguration->getDefaultProfileId($entity_type_id, $bundle_id);
        if ($config_profile === $profile->id()) {
          $used |= StrakerTranslateProfileUsageInterface::USED_BY_SETTINGS;
        }
      }
    }

    return $used;
  }

}
