<?php

namespace Drupal\straker_translate\Views;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Sql\SqlEntityStorageInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\views\EntityViewsData;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the straker_translate views integration.
 *
 * @internal
 */
class StrakerTranslateContentMetadataViewsData extends EntityViewsData {

  /**
   * The straker_translate configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translateConfigService;

  /**
   * Constructs an EntityViewsData object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type to provide views integration for.
   * @param \Drupal\Core\Entity\Sql\SqlEntityStorageInterface $storage_controller
   *   The storage handler used for this entity type.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation_manager
   *   The translation manager.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The straker_translate configuration service.
   */
  public function __construct(EntityTypeInterface $entity_type, SqlEntityStorageInterface $storage_controller, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, TranslationInterface $translation_manager, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, EntityFieldManagerInterface $entity_field_manager = NULL) {
    parent::__construct($entity_type, $storage_controller, $entity_type_manager, $module_handler, $translation_manager, $entity_field_manager);
    $this->straker_translateConfigService = $straker_translate_configuration;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('string_translation'),
      $container->get('straker_translate.configuration'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * Returns the views data.
   *
   * @return array
   *   The views data.
   */
  public function getViewsData() {
    $data = parent::getViewsData();

    $data['straker_translate_metadata']['translation_source'] = [
      'title' => $this->t('Straker translation source'),
      'description' => $this->t('The source language from which this translation was created.'),
      'field' => [
        'id' => 'field',
        'default_formatter' => 'straker_translate_translation_status',
        'field_name' => 'translation_source',
      ],
    ];

    $enabled_entity_types = array_filter($this->entityTypeManager->getDefinitions(), function (EntityTypeInterface $type) {
      return $this->straker_translateConfigService->isEnabled($type->id());
    });

    // Provides a relationship from the entity to its straker_translate status metadata
    // entity.
    $straker_translate_state_entity_type = $this->entityTypeManager->getDefinition('straker_content_metadata');
    $straker_translate_state_entity_base_table = $straker_translate_state_entity_type->getDataTable() ?: $straker_translate_state_entity_type->getBaseTable();

    foreach ($enabled_entity_types as $entity_type_id => $entity_type) {
      $table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();

      $data[$table]['straker_translate_translation_status'] = [
        'title' => t('Straker translation status'),
        'relationship' => [
          'id' => 'standard',
          'label' => $this->t('@label Straker translation status', ['@label' => $entity_type->getLabel()]),
          'base' => $straker_translate_state_entity_base_table,
          'base field' => 'content_entity_id',
          'relationship field' => $entity_type->getKey('id'),
          'extra' => [
            [
              'field' => 'content_entity_type_id',
              'value' => $entity_type_id,
            ],
          ],
        ],
        'field' => [
          'id' => 'field',
          'default_formatter' => 'straker_translate_translation_status',
          'field_name' => 'straker_translate_translation_status',
        ],
        'filter' => [
          'id' => 'straker_translate_translation_status',
          'allow empty' => TRUE,
        ],
      ];
    }

    return $data;
  }

}
