<?php

namespace Drupal\Tests\lingotek\Functional\Controller;

use Drupal\Core\Url;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\language\Entity\ContentLanguageSettings;
use Drupal\lingotek\Lingotek;
use Drupal\node\Entity\Node;
use Drupal\Tests\lingotek\Functional\LingotekTestBase;
use Drupal\Tests\system\Functional\Cache\AssertPageCacheContextsAndTagsTrait;

/**
 * Tests the notification controller.
 *
 * @group lingotek
 */
class LingotekNotificationControllerTest extends LingotekTestBase {

  use AssertPageCacheContextsAndTagsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['block', 'node', 'page_cache', 'dynamic_page_cache', 'big_pipe'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Place the actions and title block.
    $this->drupalPlaceBlock('page_title_block', ['region' => 'content', 'weight' => -5]);
    $this->drupalPlaceBlock('local_tasks_block', ['region' => 'content', 'weight' => -10]);

    // Create Article node types.
    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
    ]);

    // Add a language.
    ConfigurableLanguage::createFromLangcode('es')->setThirdPartySetting('lingotek', 'locale', 'es_AR')->save();

    // Enable translation for the current entity type and ensure the change is
    // picked up.
    ContentLanguageSettings::loadByEntityTypeBundle('node', 'article')->setLanguageAlterable(TRUE)->save();
    \Drupal::service('content_translation.manager')->setEnabled('node', 'article', TRUE);

    drupal_static_reset();
    \Drupal::entityTypeManager()->clearCachedDefinitions();
    $this->applyEntityUpdates();
    // Rebuild the container so that the new languages are picked up by services
    // that hold a list of languages.
    $this->rebuildContainer();

    $this->saveLingotekContentTranslationSettingsForNodeTypes();

    $this->enablePageCaching();
  }

  /**
   * Tests that a callback is never cached.
   */
  public function testCallbackNotCached() {
    // Login as admin.
    $this->drupalLogin($this->rootUser);

    // Create a node.
    $edit = [];
    $edit['title[0][value]'] = 'Llamas are cool';
    $edit['body[0][value]'] = 'Llamas are very cool';
    $edit['langcode[0][value]'] = 'en';
    $edit['lingotek_translation_management[lingotek_translation_profile]'] = 'automatic';
    $this->saveAndPublishNodeForm($edit);

    /** @var \Drupal\node\NodeInterface $node */
    $node = Node::load(1);
    /** @var \Drupal\lingotek\LingotekContentTranslationServiceInterface $content_translation_service */
    $content_translation_service = \Drupal::service('lingotek.content_translation');

    // Assert the content is importing.
    $this->assertSame(Lingotek::STATUS_IMPORTING, $content_translation_service->getSourceStatus($node));

    $this->goToContentBulkManagementForm();

    // Simulate the notification of content successfully uploaded.
    $url = Url::fromRoute('lingotek.notify', [], [
      'query' => [
        'project_id' => 'test_project',
        'document_id' => 'dummy-document-hash-id',
        'complete' => 'false',
        'type' => 'document_uploaded',
        'progress' => '0',
      ],
    ])->setAbsolute()->toString();
    $request = $this->client->get($url, [
      'headers' => [
        'Accept' => 'application/json',
        'Content-Type' => 'application/json',
      ],
      'http_errors' => FALSE,
    ]);
    $cache_control_header = $request->getHeader('Cache-Control');
    $this->assertStringContainsString('max-age=0', $cache_control_header[0]);

    $response = json_decode($request->getBody(), TRUE);
    dump($request);
    $this->assertSame(['es'], $response['result']['request_translations'], 'Spanish language has been requested after notification automatically.');

    // Simulate again the notification of content successfully uploaded.
    $request = $this->client->get($url, [
      'headers' => [
        'Accept' => 'application/json',
        'Content-Type' => 'application/json',
      ],
      'http_errors' => FALSE,
    ]);
    $cache_control_header = $request->getHeader('Cache-Control');
    $this->assertStringContainsString('max-age=0', $cache_control_header[0]);

    $response = json_decode($request->getBody(), TRUE);
    dump($request);
    $this->assertSame(['es'], $response['result']['request_translations'], 'Spanish language has been requested after notification automatically.');
  }

  /**
   * Tests that a callback is never cached.
   */
  public function testUnprocessedCallbackNotCached() {

    // Simulate the notification of content successfully uploaded.
    $url = Url::fromRoute('lingotek.notify', [], [
      'query' => [
        'llamas' => 'cool',
      ],
    ])->setAbsolute()->toString();
    $request = $this->client->get($url, [
      'headers' => [
        'Accept' => 'application/json',
        'Content-Type' => 'application/json',
      ],
      'http_errors' => FALSE,
    ]);
    $cache_control_header = $request->getHeader('Cache-Control');
    $this->assertStringContainsString('max-age=0', $cache_control_header[0]);

    $response = (string) $request->getBody();
    $this->assertSame($response, 'It works, but nothing to look here.');

    // Simulate again the notification of content successfully uploaded.
    $request = $this->client->get($url, [
      'headers' => [
        'Accept' => 'application/json',
        'Content-Type' => 'application/json',
      ],
      'http_errors' => FALSE,
    ]);
    $cache_control_header = $request->getHeader('Cache-Control');
    $this->assertStringContainsString('max-age=0', $cache_control_header[0]);

    $response = (string) $request->getBody();
    $this->assertSame($response, 'It works, but nothing to look here.');
  }

}
