# Quick Start Guide

This guide will help you get started with the String module quickly. We'll cover the essential concepts and show you how to use the module in your Drupal project.

## Basic Concepts

The String module introduces a new way to handle translatable strings in Drupal:

1. **String Identifiers**: Instead of using the actual text, you use unique identifiers
2. **Namespaces**: Strings are organized in namespaces (typically your module name)
3. **Groups**: Strings can be grouped using dot notation
4. **Context**: Strings can have different translations based on context

## Your First String

1. Create a new file in your custom module called `your_module.string.yml`:

```yaml
# Basic string
your_module.welcome:
  default: "Welcome to our site!"

# String with placeholder
your_module.greeting:
  default: "Hello, @name!"

# Plural string
your_module.items:
  default: "@count item"
  default_plural: "@count items"
```

2. Use the string in your code:

```php
<?php
// Basic string
$output = t('your_module.welcome');

// String with placeholder
$output = t('your_module.greeting', ['@name' => $user->getDisplayName()]);

// Plural string
$output = format_plural($count, 'your_module.items', 'your_module.items');
```

## String Management

!!!note
    Managing String Translation can be done via Drupal Core's interface translation.

### Viewing Strings

1. Go to Configuration → Regional and language → String management
2. Or visit `/admin/config/regional/string/manage`

### Exporting Strings

1. Go to Configuration → Regional and language → String export
2. Or visit `/admin/config/regional/string/export`
3. Select the strings you want to export
4. Choose the export format
5. Click "Export"

### Importing Translations

1. Go to Configuration → Regional and language → Import
2. Or visit `/admin/config/regional/translate/import`
3. Upload your translation file
4. Click "Import"

## Best Practices

1. **Use Meaningful Identifiers**:
    ```yaml
    # Good
    user.profile.welcome:
      default: "Welcome to your profile"

    # Bad
    str1: 
      default: "Welcome to your profile"
    ```

2. **Group Related Strings**:
    ```yaml
    # Good
    user.profile.welcome:
      default: "Welcome"
    user.profile.edit:
      default:  "Edit Profile"
    user.profile.delete:
      default:  "Delete Profile"

    # Bad
    welcome: 
      default: "Welcome"
    edit_profile: 
      default: "Edit Profile"
    delete_profile: 
      default: "Delete Profile"
    ```

3. **Use Context When Needed**:
!!! tip
    In most cases, when you use "group" as described in previous step, context might not be needed.
    avoiding un-necessary complications during usage.

    For example, it is easier to use `t('forms.button.submit')` over `t('button.submit', [], ['context' => 'form_button']);`

   ```yaml
   # Same string, different contexts
   button.submit:
     default: "Submit"
     msgctxt: "form_button"
   
   link.submit:
     default: "Submit"
     msgctxt: "navigation_link"
   ```

## Common Tasks

### Adding a New String

1. Add the string to your `.string.yml` file
2. Clear the cache:
   ```bash
   drush cr
   ```

### Updating a String

1. Update the string in your `.string.yml` file
2. Clear the cache:
   ```bash
   drush cr
   ```

### Translating a String

1. Export the strings
2. Translate them in your preferred editor
3. Import the translations

## Next Steps

- Check out [Best Practices](../reference/best-practices.md)