# Basic Usage Guide

This guide covers the fundamental usage of the String module in Drupal projects.

## String Definition

### Basic String Structure

```yaml
# In your_module.string.yml
your_module.welcome:
  default: "Welcome to our site!"
  comments:
    - "Displayed on the homepage"
    - "Used in the main navigation"

your_module.actions.save:
  default: "Save"
  msgctxt: "form_button"
  comments:
    - "Used in form submit buttons"
```

### String with Placeholders

```yaml
# In your_module.string.yml
your_module.user.greeting:
  default: "Hello, @name! Welcome to @site."
  comments:
    - "Displayed in user profile"
    - "Placeholders: @name (user name), @site (site name)"

your_module.notification.count:
  default: "@count new notification"
  default_plural: "@count new notifications"
  comments:
    - "Displayed in notification center"
```

## Using Strings in Code

### PHP Usage

```php
<?php

// Basic string
$output = t('your_module.welcome');

// String with placeholders
$output = t('your_module.user.greeting', [
  '@name' => $user->getDisplayName(),
  '@site' => \Drupal::config('system.site')->get('name'),
]);

// Plural string
$count = 5;
$output = format_plural($count, 'your_module.notification.count', 'your_module.notification.count');

// String with context
$output = t('your_module.actions.save', [], ['context' => 'form_button']);
```

### Twig Template Usage

```twig
{# Basic string #}
{{ 'your_module.welcome'|t }}

{# String with placeholders #}
{{ 'your_module.user.greeting'|t({
  '@name': user.display_name,
  '@site': site_name
}) }}

{# String with context #}
{{ 'your_module.actions.save'|t({}, {'context': 'form_button'}) }}
```

!!! warning
    There is no equivalent usage of `format_plural` in twig template. Hence, Following won't work.
    ```twig
    {# Plural string #}
    {{ 'your_module.notification.count'|format_plural(count) }}
    ```

### JavaScript Usage

```javascript
// Basic string
Drupal.t('your_module.welcome');

// String with placeholders
Drupal.t('your_module.user.greeting', {
  '@name': userName,
  '@site': siteName
});

// Plural string
Drupal.formatPlural(count, 'your_module.notification.count', 'your_module.notification.count');

// String with context
Drupal.t('your_module.actions.save', {}, {context: 'form_button'});
```

## Common Use Cases

### 1. Form Labels and Buttons

```yaml
# In your_module.string.yml
your_module.form:
  title:
    default: "Contact Form"
  fields:
    name:
      label:
        default: "Your Name"
      placeholder:
        default: "Enter your full name"
    email:
      label:
        default: "Email Address"
      placeholder:
        default: "Enter your email"
  buttons:
    submit:
      default: "Send Message"
    reset:
      default: "Clear Form"
```

### 2. Error Messages

```yaml
# In your_module.string.yml
your_module.errors:
  required:
    default: "@field is required"
  invalid:
    email:
      default: "Please enter a valid email address"
    phone:
      default: "Please enter a valid phone number"
  validation:
    password:
      too_short:
        default: "Password must be at least @length characters"
      too_simple:
        default: "Password is too simple"
```

### 3. Status Messages

```yaml
# In your_module.string.yml
your_module.status:
  success:
    save:
      default: "@type has been saved successfully"
    delete:
      default: "@type has been deleted successfully"
  error:
    save:
      default: "Failed to save @type"
    delete:
      default: "Failed to delete @type"
```

### 4. Navigation Items

```yaml
# In your_module.string.yml
your_module.navigation:
  home:
    default: "Home"
  profile:
    default: "My Profile"
  settings:
    default: "Settings"
  logout:
    default: "Log Out"
```

## Best Practices

1. **Consistent Naming**:
   ```yaml
   # Good
   your_module.user.profile.title
   your_module.user.profile.description
   
   # Bad
   your_module.profile_title
   your_module.profile_desc
   ```

2. **Meaningful Groups**:
   ```yaml
   # Good
   your_module.user.profile.edit
   your_module.user.profile.view
   
   # Bad
   your_module.edit_profile
   your_module.view_profile
   ```

3. **Clear Context**:
   ```yaml
   # Good
   your_module.actions.save:
     default: "Save"
     msgctxt: "form_button"
   
   your_module.navigation.save:
     default: "Save"
     msgctxt: "menu_item"
   ```
!!! note "About using context"

    Since strings are now identified with unique unique identifiers, this means
    in most cases "context" becomes un-necessary.

4. **Helpful Comments**:
   ```yaml
   your_module.user.welcome:
     default: "Welcome back!"
     comments:
       - "Displayed in user dashboard"
       - "Shown after successful login"
       - "Used in user greeting block"
   ```

## Next Steps

- Check out [Best Practices](../reference/best-practices.md)