# String i18next

A Drupal module that provides i18next integration for the String module, enabling
seamless translation management for JavaScript applications.

## Overview

The String i18next module extends the functionality of the String module by
providing integration with i18next, a popular internationalization framework for
JavaScript. This integration allows you to use the same string definitions from
your Drupal site in your JavaScript applications, maintaining consistency across
your entire application stack.

## Features

- Seamless integration between Drupal's String module and i18next
- Automatic conversion of String module's YAML definitions to i18next format
- Support for both single and plural string forms
- Maintains context information for accurate translations
- Compatible with i18next's namespace structure
- Supports all String module's string definition formats
- JSON endpoint for consuming translations in JavaScript applications

## Requirements

- Drupal 10.3 or higher
- String module

## Installation

1. Install the module using Composer:
   ```bash
   composer require drupal/string_i18next
   ```

2. Enable the module through the Drupal admin interface or using Drush:
   ```bash
   drush en string_i18next
   ```

4. Configure User permissions:
   - Go to Administration → People → Permissions
   - Enable "Access string_i18next endpoint" for anonymous users if you want
     public access
   - Or enable it for authenticated users if you want to restrict access

!!!note "Rest endpoints"
    The module exposes "rest" resources for collection as well as items.
    These can be enabled using modules like "Rest UI".


## Usage

### API Endpoints

The module provides two endpoints for consuming translations:

#### 1. All Translations
```
GET /api/string/i18next/v4/{language_code}
```
Fetches all available translations for the specified language.

Parameters:
- `language_code`: The language code to fetch translations for (e.g., 'en', 'fr',
  'de')

Response format:
```json
{
  "dashboard": {
    "welcome_message": "Welcome to your dashboard"
  },
  "search": {
    "result": {
      "items_count": {
        "pluralForm_zero": "0 items found",
        "pluralForm_one": "1 item found",
        "pluralForm_other": "{{count}} items found"
      }
    }
  }
}
```

!!!note "For string identifiers that have nested children"
    In case we have string ids `a.b` and `a.b.c`, then `a.b` is available
    at `a.b_key`

!!! note "Plural Forms" 
    Plural forms gets a suffix `.pluralForm`.

#### 2. Namespace-Specific Translations
```
GET /api/string/i18next/v4/{language_code}/{namespace}
```
Fetches translations for a specific namespace. This is useful when you only need
translations for a particular section of your application.

Parameters:
- `language_code`: The language code to fetch translations for (e.g., 'en', 'fr',
  'de')
- `namespace`: The namespace to filter translations (e.g., 'dashboard', 'search')

Example request:
```
GET /api/string/i18next/v4/en/dashboard
```

Response format:
```json
{
  "welcome_message": "Welcome to your dashboard",
  "title": "Dashboard",
  "navigation": {
    "home": "Home",
    "settings": "Settings"
  }
}
```

### Accessing Translations in JavaScript

You can initialize i18next with the translations from the REST endpoint:

```javascript
// Initialize i18next with the translations from the REST endpoint
async function initI18next() {
  // Fetch all translations
  const response = await fetch('/api/string/i18next/v4/en?_format=json');
  const translations = await response.json();

  await i18next.init({
    resources: {
      en: {
        translation: translations
      }
    }
  });
}

// Or fetch namespace-specific translations
async function initNamespaceTranslations() {
  // Fetch only dashboard translations
  const response = await fetch('/api/string/i18next/v4/en/dashboard?_format=json');
  const translations = await response.json();

  await i18next.init({
    resources: {
      en: {
        dashboard: translations  // Load as a separate namespace
      }
    }
  });
}

// Use the translations
initI18next().then(() => {
  i18next.t('dashboard.welcome_message');
  i18next.t('search.result.items_count.pluralForm', { count: 5 });
});
```

### String Format Conversion

The module automatically handles the conversion of String module's format to
i18next format:

- Namespace.key structure is preserved
- Context information is appended to the key when present
- Plural forms are converted to i18next's plural format
- Placeholders are converted to i18next's interpolation format

### Integration with String Module

The String i18next module works seamlessly with the String module's features:

- Supports all string definition formats from `.string.yml` files
- Maintains the namespace.key structure for string identifiers
- Preserves context information for accurate translations
- Supports both single and plural string forms

## Example String Definition

```yaml
# In your module's .string.yml file
dashboard.welcome_message:
  default: "Welcome to your dashboard"
  msgctxt: "dashboard"

search.result.items_count:
  default: "@count item found"
  default_plural: "@count items found"
```

This will be available via the REST endpoint as:

```json
{
  "dashboard": {
    "welcome_message": "Welcome to your dashboard"
  },
  "search": {
    "result": {
      "items_count": {
        "pluralForm_zero": "0 items found",
        "pluralForm_one": "1 item found",
        "pluralForm_other": "{{count}} items found"
      }
    }
  }
}
```

## Caching

The REST endpoint responses are cached for better performance. You can clear the
cache using:

```bash
drush cr
```

## Related Modules

- [String](https://www.drupal.org/project/string) - Base module for string management
- [i18next](https://www.i18next.com/) - JavaScript internationalization framework
- [REST UI](https://www.drupal.org/project/restui) - A user interface for configuring Drupal REST module.

## Support

For issues and feature requests, please use the [issue queue](https://www.drupal.org/project/issues/string) on Drupal.org.
