<?php

declare(strict_types=1);

namespace Drupal\string_i18next\Controller;

use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\string_i18next\StringI18Next;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Returns responses for String i18next routes.
 */
class StringI18nextController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly StringI18Next $stringI18next,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('string_i18next'),
    );
  }

  /**
   * Returns the collection of strings for a language code.
   */
  public function collection(string $language_code): JsonResponse {
    $data = $this->stringI18next->getStrings($language_code);
    return $this->generateCacheableResponse($data);
  }

  /**
   * Returns the string for an id.
   */
  public function item(string $language_code, string $id): JsonResponse {
    $items = explode('.', $id);
    $strings = $this->stringI18next->getStrings($language_code);
    $filtered_strings = $strings;
    foreach ($items as $key) {
      if (isset($filtered_strings[$key])) {
        $filtered_strings = $filtered_strings[$key];
      }
      else {
        throw new NotFoundHttpException();
      }
    }
    return $this->generateCacheableResponse($filtered_strings);
  }

  /**
   * Generates a cacheable JSON response.
   */
  protected function generateCacheableResponse(array $data): CacheableJsonResponse {
    $data['#cache'] = [
      'contexts' => [
        'url',
      ],
      'tags' => [
        'string',
        'config:core.extension',
      ],
    ];
    $response = new CacheableJsonResponse($data);
    $response->addCacheableDependency(CacheableMetadata::createFromRenderArray($data));
    return $response;
  }

}
