<?php

namespace Drupal\string_i18next;

use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\locale\StringStorageInterface;
use Drupal\string\DTO\StringDefinition;
use Drupal\string\StringManager;

/**
 * StringI18Next class definition.
 */
class StringI18Next {

  use StringTranslationTrait;

  /**
   * A large number used for plural form testing.
   */
  private const PLURAL_TEST_COUNT = 111222333444;

  /**
   * The string manager service.
   *
   * @var \Drupal\string\StringManagerInterface
   */
  protected $stringManager;

  /**
   * The local storage service.
   *
   * @var \Drupal\locale\StringStorageInterface
   */
  protected $localStorage;

  /**
   * The constructor.
   */
  public function __construct(StringManager $string_manager, StringStorageInterface $local_storage) {
    $this->stringManager = $string_manager;
    $this->localStorage = $local_storage;
  }

  /**
   * Inject parent conflict resolver context into the definitions.
   *
   * This method handles cases where a parent key conflicts with
   * a child key in the i18next format.
   * For example, if we have 'a.b' and 'a.b.c',
   * we need to modify 'a.b' to be 'a.b_key' to avoid conflicts.
   *
   * @param array &$definitions
   *   The string definitions array to modify.
   */
  protected function injectParentConflictResolverContext(array &$definitions): void {
    $keys = array_keys($definitions);
    sort($keys, SORT_STRING);

    $previous_key = NULL;
    foreach ($keys as $key) {
      if ($previous_key && $this->isParentKey($previous_key, $key)) {
        $this->appendContextToDefinition($definitions[$previous_key]);
      }
      $previous_key = $key;
    }
  }

  /**
   * Check if the previous key is a parent of the current key.
   *
   * @param string $previous_key
   *   The previous key to check.
   * @param string $current_key
   *   The current key to check against.
   *
   * @return bool
   *   TRUE if the previous key is a parent of the current key.
   */
  private function isParentKey(string $previous_key, string $current_key): bool {
    return str_starts_with($current_key, $previous_key . '.');
  }

  /**
   * Append context to a definition to avoid key conflicts.
   *
   * @param array &$definition
   *   The definition to modify.
   */
  private function appendContextToDefinition(array &$definition): void {
    $context_key = StringDefinition::MSG_CONTEXT;
    if (!empty($definition[$context_key])) {
      $definition[$context_key] .= '_key';
    }
    else {
      $definition[$context_key] = 'key';
    }
  }

  /**
   * Get strings for a given language code.
   */
  public function getStrings($language_code) {
    $definitions = $this->stringManager->getDefinitions();
    $this->injectParentConflictResolverContext($definitions);
    $output = [];
    foreach ($definitions as $key => $value) {
      $definition = new StringDefinition($value);
      $key = $this->formatKeyForI18NextItem($key, $definition);
      $items = explode('.', $key);
      $leaf = NULL;
      foreach ($items as $k) {
        if ($leaf !== NULL) {
          if (is_string($leaf)) {
            $leaf = [];
          }
          $leaf = &$leaf[$k];
        }
        else {
          $leaf = &$output[$k];
        }
        $leaf = $leaf ?? [];
      }
      $leaf = $this->convertPluginItemToI18NextItem($definition, $language_code);
      unset($leaf);
    }
    return $output;
  }

  /**
   * Create keys compatible with i18next.
   */
  protected function formatKeyForI18NextItem($key, StringDefinition $definition) {
    $key = str_replace(':', '.', $key);
    if ($definition->getContext()) {
      // Use "_" (i.e. underscore) to specific context.
      // @see https://www.i18next.com/translation-function/context
      $key .= '_' . str_replace(' ', '_', $definition->getContext());
    }
    return $key;
  }

  /**
   * Convert string definition to i18next friendly definition.
   */
  protected function convertPluginItemToI18NextItem(StringDefinition $definition, $language_code) {
    $context = $definition->getContext();
    $options = [
      'context' => $context,
    ];
    if ($language_code !== 'automatic') {
      $options['langcode'] = $language_code;
    }
    foreach ($definition->getPlaceholders() as $placeholder) {
      $placeholder_key = $placeholder['key'];
      // Convert '@placeholder' => '{{placeholder}}'.
      $placeholders[$placeholder_key] = '{{' . substr($placeholder_key, '1') . '}}';
    }
    $placeholders['@count'] = '{{count}}';
    if ($definition->getDefaultValuePlural()) {
      return $this->getPluralForm($definition, $placeholders, $options);
    }
    else {
      return $this->getSingleForm($definition, $placeholders, $options);
    }
  }

  /**
   * Get single form.
   */
  protected function getSingleForm(StringDefinition $definition, $placeholders, $options): string {
    // phpcs:ignore Drupal.Semantics.FunctionT -- Dynamic string ID is required.
    return $this->t($definition->getStringId(), $placeholders, $options)->render();
  }

  /**
   * Get plural form.
   */
  protected function getPluralForm(StringDefinition $definition, $placeholders, $options): array {
    $id = $definition->getStringId();
    $zero = new PluralTranslatableMarkup(0, $id, $id, $placeholders, $options);
    $one = new PluralTranslatableMarkup(1, $id, $id, $placeholders, $options);
    // A rare case where we inject actual count as placeholder
    // and replace that with the placeholder ¯\_(ツ)_/¯.
    $other = new PluralTranslatableMarkup(self::PLURAL_TEST_COUNT, $id, $id, $placeholders, $options);
    return [
      'pluralForm_zero' => $zero->render(),
      'pluralForm_one' => $one->render(),
      'pluralForm_other' => str_replace((string) self::PLURAL_TEST_COUNT, '{{count}}', $other->render()),
    ];
  }

}
