<?php

namespace Drupal\Tests\string_i18next\Unit;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\locale\StringStorageInterface;
use Drupal\string\StringManager;
use Drupal\string_i18next\StringI18Next;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * Unit tests for StringI18Next class.
 *
 * @coversDefaultClass \Drupal\string_i18next\StringI18Next
 * @group string_i18next
 */
class StringI18NextTest extends UnitTestCase {

  use ProphecyTrait;
  use StringTranslationTrait;

  /**
   * The string manager service.
   *
   * @var \Drupal\string\StringManager|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringManager;

  /**
   * The local storage service.
   *
   * @var \Drupal\locale\StringStorageInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $localStorage;

  /**
   * The string i18next service.
   *
   * @var \Drupal\string_i18next\StringI18Next
   */
  protected $stringI18Next;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->stringManager = $this->prophesize(StringManager::class);
    $this->localStorage = $this->prophesize(StringStorageInterface::class);

    $this->stringI18Next = new StringI18Next(
      $this->stringManager->reveal(),
      $this->localStorage->reveal()
    );
  }

  /**
   * @covers ::getStrings
   */
  public function testGetStrings(): void {
    // Mock the string definitions that would be returned by StringManager.
    $definitions = [
      'test.simple' => [
        'id' => 'test.simple',
        'string_id' => 'test.simple',
        'default' => 'Simple string',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.with_context' => [
        'id' => 'test.with_context',
        'string_id' => 'test.with_context',
        'default' => 'String with context',
        'msgctxt' => 'test_context',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.plural' => [
        'id' => 'test.plural',
        'string_id' => 'test.plural',
        'default' => '@count item',
        'default_plural' => '@count items',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
    ];

    // Configure the string manager mock to return our test definitions.
    $this->stringManager->getDefinitions()
      ->willReturn($definitions);

    // Mock the getSingleForm and getPluralForm methods.
    $stringI18Next = $this->getMockBuilder(StringI18Next::class)
      ->setConstructorArgs([
        $this->stringManager->reveal(),
        $this->localStorage->reveal(),
      ])
      ->onlyMethods(['getSingleForm', 'getPluralForm'])
      ->getMock();

    // Configure the mock to return predefined values for each string.
    $stringI18Next->expects($this->exactly(2))
      ->method('getSingleForm')
      ->willReturnCallback(function ($definition, $language_code) {
        return $definition->getDefaultValue();
      });

    $stringI18Next->expects($this->exactly(1))
      ->method('getPluralForm')
      ->willReturnCallback(function ($definition, $language_code) {
        return [
          'zero' => '0',
          'one' => '1',
          'other' => '{{count}}',
        ];
      });

    // Test with English language code.
    $result = $stringI18Next->getStrings('en');

    // Assert the expected structure of the result.
    $this->assertEquals([
      'test' => [
        'simple' => 'Simple string',
        'with_context_test_context' => 'String with context',
        'plural' => [
          'zero' => '0',
          'one' => '1',
          'other' => '{{count}}',
        ],
      ],
    ], $result);
  }

  /**
   * @covers ::getStrings
   */
  public function testGetStringsWithNestedKeys(): void {
    // Mock the string definitions with nested keys.
    $definitions = [
      'test.nested.key1' => [
        'id' => 'test.nested.key1',
        'string_id' => 'test.nested.key1',
        'default' => 'Nested key 1',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nested.key2' => [
        'id' => 'test.nested.key2',
        'string_id' => 'test.nested.key2',
        'default' => 'Nested key 2',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nested.key3' => [
        'id' => 'test.nested.key3',
        'string_id' => 'test.nested.key3',
        'default' => '',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
    ];

    // Configure the string manager mock.
    $this->stringManager->getDefinitions()
      ->willReturn($definitions);

    // Mock the getSingleForm method.
    $stringI18Next = $this->getMockBuilder(StringI18Next::class)
      ->setConstructorArgs([
        $this->stringManager->reveal(),
        $this->localStorage->reveal(),
      ])
      ->onlyMethods(['getSingleForm'])
      ->getMock();

    // Configure the mock to return predefined values.
    $stringI18Next->expects($this->exactly(3))
      ->method('getSingleForm')
      ->willReturnCallback(function ($definition, $language_code) {
        return $definition->getDefaultValue();
      });

    // Test with English language code.
    $result = $stringI18Next->getStrings('en');

    // Assert the expected nested structure.
    $this->assertEquals([
      'test' => [
        'nested' => [
          'key1' => 'Nested key 1',
          'key2' => 'Nested key 2',
          'key3' => '',
        ],
      ],
    ], $result);
  }

  /**
   * @covers ::getStrings
   */
  public function testGetStringsWithSpecialNestedKeys(): void {
    // Mock the string definitions with nested keys.
    $definitions = [
      'test.nested' => [
        'id' => 'test.nested',
        'string_id' => 'test.nested',
        'default' => 'Nested key 1',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nested.key2' => [
        'id' => 'test.nested.key2',
        'string_id' => 'test.nested.key2',
        'default' => 'Nested key 2',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nested.key3' => [
        'id' => 'test.nested.key3',
        'string_id' => 'test.nested.key3',
        'default' => '',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nestedWithContext' => [
        'id' => 'test.nested',
        'string_id' => 'test.nested',
        'default' => 'Nested key 1',
        'msgctxt' => 'someContext',
        'provider' => 'test',
        'getLocations' => [],
      ],
      'test.nestedWithContext.key2' => [
        'id' => 'test.nested.key2',
        'string_id' => 'test.nested.key2',
        'default' => 'Nested key 2',
        'msgctxt' => '',
        'provider' => 'test',
        'getLocations' => [],
      ],
    ];

    // Configure the string manager mock.
    $this->stringManager->getDefinitions()
      ->willReturn($definitions);

    // Mock the getSingleForm method.
    $stringI18Next = $this->getMockBuilder(StringI18Next::class)
      ->setConstructorArgs([
        $this->stringManager->reveal(),
        $this->localStorage->reveal(),
      ])
      ->onlyMethods(['getSingleForm'])
      ->getMock();

    // Configure the mock to return predefined values.
    $stringI18Next->expects($this->exactly(5))
      ->method('getSingleForm')
      ->willReturnCallback(function ($definition, $language_code) {
        return $definition->getDefaultValue();
      });

    // Test with English language code.
    $result = $stringI18Next->getStrings('en');

    // Assert the expected nested structure.
    $this->assertEquals([
      'test' => [
        'nested_key' => 'Nested key 1',
        'nestedWithContext_someContext_key' => 'Nested key 1',
        'nested' => [
          'key2' => 'Nested key 2',
          'key3' => '',
        ],
        'nestedWithContext' => [
          'key2' => 'Nested key 2',
        ],
      ],
    ], $result);
  }

}
