<?php

namespace Drupal\string\DTO;

use Drupal\string\Exception\InvalidStringDefinitionException;

/**
 * Data Transfer Object for String Definition.
 *
 * This class encapsulates all properties and constants
 * related to string definitions used in the
 * StringManager class.
 */
class StringDefinition {

  /**
   * Message ID constant.
   */
  public const MSG_ID = 'msgid';

  /**
   * Plural message ID constant.
   */
  public const MSG_ID_PLURAL = 'msgid_plural';

  /**
   * Message context constant.
   */
  public const MSG_CONTEXT = 'msgctxt';

  /**
   * Translations constant.
   */
  public const TRANSLATIONS = 'translations';

  /**
   * ID constant.
   */
  public const ID = 'id';

  /**
   * String ID constant.
   */
  public const STRING_ID = 'sid';

  /**
   * Locale ID constant.
   */
  public const LOCALE_ID = 'lid';

  /**
   * Default value constant.
   */
  public const DEFAULT_VALUE = 'default';

  /**
   * Default plural value constant.
   */
  public const DEFAULT_VALUE_PLURAL = 'default_plural';

  /**
   * Title constant.
   */
  public const TITLE = 'title';

  /**
   * Description constant.
   */
  public const DESCRIPTION = 'description';

  /**
   * Comments constant.
   */
  public const COMMENTS = 'comments';

  public const PLACEHOLDERS = 'placeholders';


  /**
   * The string identifier.
   *
   * @var string
   */
  private string $id;

  /**
   * The string ID.
   *
   * @var string
   */
  private string $stringId;

  /**
   * The locale ID.
   *
   * @var int
   */
  private int $localeId;

  /**
   * The default value.
   *
   * @var string
   */
  private string $defaultValue;

  /**
   * The default plural value.
   *
   * @var string|null
   */
  private ?string $defaultValuePlural;

  /**
   * The title.
   *
   * @var string|null
   */
  private ?string $title;

  /**
   * The description.
   *
   * @var string|null
   */
  private ?string $description;

  /**
   * The comments.
   *
   * @var string[]|null
   */
  private ?array $comments;

  /**
   * The message context.
   *
   * @var string
   */
  private string $context;

  /**
   * The provider.
   *
   * @var string
   */
  private string $provider;

  /**
   * The locations.
   *
   * @var array
   */
  private array $locations;

  /**
   * The translations.
   *
   * @var array
   */
  private ?array $translations;

  /**
   * The placeholders.
   *
   * @var array
   */
  private array $placeholders;

  /**
   * Constructs a new StringDefinition object.
   *
   * @param array $definition
   *   The definition array containing string properties.
   */
  public function __construct(array $definition) {
    $this->id = $definition[self::ID] ?? '';
    $this->stringId = $definition[self::STRING_ID] ?? '';
    $this->localeId = $definition[self::LOCALE_ID] ?? 0;
    $this->defaultValue = $definition[self::DEFAULT_VALUE] ?? '';
    $this->defaultValuePlural = $definition[self::DEFAULT_VALUE_PLURAL] ?? NULL;
    $this->title = $definition[self::TITLE] ?? NULL;
    $this->description = $definition[self::DESCRIPTION] ?? NULL;
    $this->comments = $definition[self::COMMENTS] ?? NULL;
    $this->context = $definition[self::MSG_CONTEXT] ?? '';
    $this->translations = $definition[self::TRANSLATIONS] ?? NULL;
    $this->placeholders = $definition[self::PLACEHOLDERS] ?? [];
    $this->provider = $definition['provider'] ?? '';
    $this->locations = $definition['getLocations'] ?? [];
    if (!$this->hasValidNamespace()) {
      throw new InvalidStringDefinitionException("Plugin 'id' must contain format 'namespace.key'. The id '{$this->id}' is invalid");
    }
    if (!$this->hasValidId()) {
      throw new InvalidStringDefinitionException('Plugin property "id" is required for string plugin definition.');
    }
  }

  /**
   * Gets the string identifier.
   *
   * @return string
   *   The string identifier.
   */
  public function getId(): string {
    return $this->id;
  }

  /**
   * Gets the string ID.
   *
   * @return string
   *   The string ID.
   */
  public function getStringId(): string {
    return $this->stringId;
  }

  /**
   * Gets the locale ID.
   *
   * @return int
   *   The locale ID.
   */
  public function getLocaleId(): int {
    return $this->localeId;
  }

  /**
   * Gets the default value.
   *
   * @return string
   *   The default value.
   */
  public function getDefaultValue(): string {
    return $this->defaultValue;
  }

  /**
   * Gets the default plural value.
   *
   * @return string|null
   *   The default plural value.
   */
  public function getDefaultValuePlural(): ?string {
    return $this->defaultValuePlural;
  }

  /**
   * Gets the title.
   *
   * @return string|null
   *   The title.
   */
  public function getTitle(): ?string {
    return $this->title;
  }

  /**
   * Gets the description.
   *
   * @return string|null
   *   The description.
   */
  public function getDescription(): ?string {
    return $this->description;
  }

  /**
   * Gets the comments.
   *
   * @return string[]|null
   *   The comments.
   */
  public function getComments(): ?array {
    return $this->comments;
  }

  /**
   * Gets the message context.
   *
   * @return string
   *   The message context.
   */
  public function getContext(): string {
    return $this->context;
  }

  /**
   * Gets the provider.
   *
   * @return string
   *   The provider.
   */
  public function getProvider(): string {
    return $this->provider;
  }

  /**
   * Gets the locations.
   *
   * @return array
   *   The locations.
   */
  public function getLocations(): array {
    return $this->locations;
  }

  /**
   * Gets the translations.
   *
   * @return array|null
   *   The translations.
   */
  public function getTranslations(): ?array {
    return $this->translations;
  }

  /**
   * Gets the placeholders.
   *
   * @return array
   *   The placeholders.
   */
  public function getPlaceholders(): array {
    return $this->placeholders;
  }

  /**
   * Converts the DTO to an array.
   *
   * @return array
   *   The array representation of the DTO.
   */
  public function toArray(): array {
    return [
      self::ID => $this->id,
      self::STRING_ID => $this->stringId,
      self::LOCALE_ID => $this->localeId,
      self::DEFAULT_VALUE => $this->defaultValue,
      self::DEFAULT_VALUE_PLURAL => $this->defaultValuePlural,
      self::TITLE => $this->title,
      self::DESCRIPTION => $this->description,
      self::COMMENTS => $this->comments,
      self::MSG_CONTEXT => $this->context,
      self::TRANSLATIONS => $this->translations,
      self::PLACEHOLDERS => $this->placeholders,
      'provider' => $this->provider,
      'getLocations' => $this->locations,
    ];
  }

  /**
   * Checks if the definition is valid.
   *
   * @return bool
   *   TRUE if the definition is valid, FALSE otherwise.
   */
  public function isValid(): bool {
    return $this->hasValidId() && $this->hasValidNamespace();
  }

  /**
   * Checks if the namespace is valid.
   *
   * @return bool
   *   TRUE if the namespace is valid, FALSE otherwise.
   */
  protected function hasValidNamespace(): bool {
    $splits = explode('.', $this->getId());
    $has_proper_namespace = count($splits) >= 2;
    return $has_proper_namespace;
  }

  /**
   * Checks if the ID is valid.
   *
   * @return bool
   *   TRUE if the ID is valid, FALSE otherwise.
   */
  protected function hasValidId(): bool {
    return !empty($this->getId());
  }

}
