<?php

namespace Drupal\string\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\Context\ContextProviderInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Drupal\string\DTO\StringDefinition;
use Drupal\string\StringManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Form for export options.
 */
class ExportForm extends FormBase {

  use StringTranslationTrait;

  /**
   * The language manager.
   */
  protected ConfigurableLanguageManagerInterface $languageManager;

  /**
   * The current language context.
   */
  protected ContextProviderInterface $languageCurrentLanguageContext;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $loggerChannelDefault;

  /**
   * The string manager.
   */
  protected StringManagerInterface $stringManager;

  /**
   * The date time service.
   */
  protected TimeInterface $dateTime;

  /**
   * The file repository.
   */
  protected FileRepositoryInterface $fileRepository;

  public function __construct(
    ConfigurableLanguageManagerInterface $languageManager,
    ContextProviderInterface $languageCurrentLanguageContext,
    LoggerChannelInterface $loggerChannelDefault,
    StringManagerInterface $stringManager,
    TimeInterface $dateTime,
    FileRepositoryInterface $fileRepository,
  ) {
    $this->languageManager = $languageManager;
    $this->languageCurrentLanguageContext = $languageCurrentLanguageContext;
    $this->loggerChannelDefault = $loggerChannelDefault;
    $this->stringManager = $stringManager;
    $this->dateTime = $dateTime;
    $this->fileRepository = $fileRepository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('language_manager'),
      $container->get('language.current_language_context'),
      $container->get('logger.channel.default'),
      $container->get('plugin.manager.string'),
      $container->get('datetime.time'),
      $container->get('file.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'string_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->configFactory()->getEditable('string.settings');
    $form['strings'] = [
      '#type' => 'tableselect',
      '#header' => [],
      '#options' => [],
      '#title' => $this->t('Table'),
    ];
    $form['show_export'] = [
      '#type' => 'checkbox',
      '#default_value' => 0,
      '#title' => $this->t('Export Pot File to machine path?'),
    ];
    $form['destination'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Interface directory'),
      '#default_value' => empty($config->get('destination.path')) ? 'public://string.pot' : $config->get('destination.path'),
      '#maxlength' => 255,
      '#required' => TRUE,
      '#description' => $this->t('A local file system path where "POT" file will be stored.'),
      '#states' => [
        'visible' => [
          ':input[name="show_export"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['actions']['download'] = [
      '#type' => 'submit',
      '#value' => $this->t('Download'),
      '#submit' => [[$this, 'submitDownloadForm']],
    ];
    $form['actions']['export'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export'),
      '#submit' => [[$this, 'submitExportForm']],
      '#states' => [
        'visible' => [
          ':input[name="show_export"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->submitDownloadForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitExportForm(array &$form, FormStateInterface $form_state): void {
    $destination = $form_state->getValue('destination');
    $this->exportFile($form_state, $destination);
  }

  /**
   * {@inheritdoc}
   */
  public function submitDownloadForm(array &$form, FormStateInterface $form_state): void {
    $this->downloadFile($form_state);
  }

  /**
   * Build pot file.
   *
   * @return string[]
   *   Build the Pot file to have necessary items and be more user friendly.
   */
  protected function buildPotFile(): array {
    $output = $this->getHeader();
    if ($definitions = $this->stringManager->getDefinitions()) {
      foreach ($definitions as $item) {
        $definition = new StringDefinition($item);
        $msgid_plural = $this->getStringMessagePluralId($definition);
        $output[] = '';
        if (!empty($definition->getComments())) {
          foreach ($definition->getComments() as $comment) {
            $output[] = '# ' . $comment;
          }
        }
        $output[] = '#. Following text is used as default if no translation is available';
        $output[] = '#. -----------------------------';
        $output[] = '#. ' . $definition->getDefaultValue();
        $output[] = '#. -----------------------------';
        if ($definition->getDefaultValue() !== NULL) {
          $output[] = '#.';
          $output[] = '#.';
          $output[] = '#. Original Drupal text that is being overridden is as follows';
          $output[] = '#. -----------------------------';
          if ($msgid_plural) {
            $output[] = '#. SINGULAR';
            $output[] = '#. ' . $definition->getDefaultValue();
            $output[] = '#. ';
            $output[] = '#. PLURAL';
            $output[] = '#. ' . $definition->getDefaultValuePlural();
          }
          else {
            $output[] = '#. ' . $definition->getDefaultValue();
          }
          $output[] = '#. -----------------------------';
        }
        $output[] = '#: ' . $this->getStringSourceId($definition);
        if ($definition->getContext() !== '') {
          $output[] = 'msgctxt "' . $definition->getContext() . '"';
        }
        $output[] = 'msgid "' . $this->getStringMessageId($definition) . '"';
        if ($msgid_plural) {
          $output[] = 'msgid_plural "' . $msgid_plural . '"';
          $output[] = 'msgstr[0] ""';
          $output[] = 'msgstr[1] ""';
        }
        else {
          $output[] = 'msgstr ""';
        }
      }
    }
    return $output;
  }

  /**
   * Get message id.
   *
   * @param \Drupal\string\StringDefinition $definition
   *   The string definition.
   *
   * @return string
   *   The message ID.
   */
  protected function getStringMessageId(StringDefinition $definition): string {
    $id = $definition->getStringId();
    if ($definition->getDefaultValue() !== NULL) {
      $id = $definition->getDefaultValue();
    }
    return $id;
  }

  /**
   * Get source id for plural form.
   *
   * @param \Drupal\string\StringDefinition $definition
   *   The string definition.
   *
   * @return string
   *   The plural message ID.
   */
  protected function getStringMessagePluralId(StringDefinition $definition): string {
    $id = '';
    if ($definition->getDefaultValuePlural() !== NULL) {
      $id = $definition->getDefaultValuePlural();
    }
    elseif ($definition->getDefaultValue() !== NULL) {
      $id = $definition->getStringId();
    }
    return $id;
  }

  /**
   * Get source id.
   *
   * @param \Drupal\string\StringDefinition $definition
   *   The string definition.
   *
   * @return string
   *   The source ID.
   */
  protected function getStringSourceId(StringDefinition $definition): string {
    return $definition->getStringId();
  }

  /**
   * Get header for pot file.
   *
   * @return string[]
   *   Header as list of string, where each item is single line in pot file.
   */
  protected function getHeader(): array {
    return [
      '# $Id$',
      '#',
      '# Export generated by string module (Drupal).',
      '# Time ' . $this->dateTime->getRequestTime(),
      '#',
      '#, fuzzy',
      'msgid ""',
      'msgstr ""',
      '"Project-Id-Version: PROJECT VERSION\n"',
      '"POT-Creation-Date: 2020-07-06 11:52+0000\n"',
      '"PO-Revision-Date: YYYY-mm-DD HH:MM+ZZZZ\n"',
      '"Last-Translator: NAME <EMAIL@ADDRESS>\n"',
      '"Language-Team: LANGUAGE <EMAIL@ADDRESS>\n"',
      '"MIME-Version: 1.0\n"',
      '"Content-Type: text/plain; charset=utf-8\n"',
      '"Content-Transfer-Encoding: 8bit\n"',
      '"Plural-Forms: nplurals=INTEGER; plural=EXPRESSION;\n"',
    ];
  }

  /**
   * Export content to file.
   *
   * @param string $content
   *   Content of file to be saved.
   * @param string $destination
   *   Destination of file to be saved.
   *
   * @return \Drupal\file\FileInterface|false
   *   File that has been exported.
   */
  protected function validateExportFile(string $content, string $destination): FileInterface|false {
    $schema = StreamWrapperManager::getScheme($destination);
    if ($schema == 'public') {
      $file = $this->fileRepository->writeData($content, $destination, FileExists::Replace);
      $this->messenger()->addStatus($this->t('The translate items has been export to %destination', [
        '%destination' => $destination,
      ]));
      return $file;
    }
    $this->messenger()->addError($this->t('Invalid stream wrapper: %destination', ['%destination' => $destination]));
    return FALSE;
  }

  /**
   * Export content to file and Create file in case it is missing.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Values from submit method.
   * @param string|null $destination
   *   Destination where file should be saved.
   */
  protected function exportFile(FormStateInterface $form_state, ?string $destination): void {
    $destination = $form_state->getValue('destination');
    if (!is_string($destination)) {
      return;
    }
    $schema = StreamWrapperManager::getScheme($destination);
    $target = StreamWrapperManager::getTarget($destination);
    if (!is_string($target)) {
      return;
    }
    $filename = substr($target, strrpos($target, '/') + 1);
    $target_without_filename = str_replace($filename, '', $target);
    $output = $this->buildPotFile();
    $this->configFactory()->getEditable('string.settings')->set('destination.path', $destination)->save();
    if (file_exists($destination)) {
      $this->validateExportFile(implode("\n", $output), $destination);
    }
    else {
      mkdir($schema . '://' . $target_without_filename, 0777, TRUE);
      $file = File::create([
        'uid' => 1,
        'filename' => basename($destination),
        'uri' => $destination,
        'status' => 1,
      ]);
      $file_uri = $file->getFileUri();
      if (is_string($file_uri)) {
        $this->validateExportFile(implode("\n", $output), $file_uri);
      }
    }
  }

  /**
   * Export file first to hardcoded destination then download the file.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Values from submit method.
   */
  protected function downloadFile(FormStateInterface $form_state): void {
    $destination = 'public://string.pot';
    $this->exportFile($form_state, $destination);
    $response = new BinaryFileResponse($destination);
    $response->setContentDisposition('attachment', 'string.pot');
    $form_state->setResponse($response);
  }

}
