<?php

namespace Drupal\string;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Plugin\Discovery\ContainerDerivativeDiscoveryDecorator;
use Drupal\Core\Plugin\Discovery\YamlDiscovery;
use Drupal\locale\StringStorageInterface;
use Drupal\string\Exception\InvalidStringDefinitionException;
use Drupal\string\DTO\StringDefinition;

/**
 * StringManager class definition.
 */
class StringManager extends DefaultPluginManager implements StringManagerInterface {

  /**
   * The local storage service.
   *
   * @var \Drupal\locale\StringStorageInterface
   */
  protected $localStorage;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Constructs a new StringManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\locale\StringStorageInterface $local_storage
   *   Locale string storage.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   The theme handler.
   */
  public function __construct(\Traversable $namespaces, ModuleHandlerInterface $module_handler, CacheBackendInterface $cache_backend, StringStorageInterface $local_storage, ThemeHandlerInterface $theme_handler) {
    $this->moduleHandler = $module_handler;
    $this->themeHandler = $theme_handler;
    $this->localStorage = $local_storage;
    $this->setCacheBackend($cache_backend, 'string', ['string', 'config:core.extension']);
    parent::__construct(FALSE, $namespaces, $module_handler);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDiscovery() {
    if ($this->discovery === NULL) {
      $extra_directories = [];
      $this->discovery = new YamlDiscovery('string', $this->moduleHandler->getModuleDirectories() + $extra_directories);
      $this->discovery = new ContainerDerivativeDiscoveryDecorator($this->discovery);
    }
    return $this->discovery;
  }

  /**
   * Check if the identifier uses proper namespace.key convention.
   *
   * @param string $id
   *   The plugin ID to check.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   If the ID format is invalid.
   */
  protected function checkForNamespace(string $id): void {
    $splits = explode('.', $id);
    if (count($splits) < 2) {
      throw new PluginException("Plugin 'id' must contain format 'namespace.key'. The id '{$id}' is invalid");
    }
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.parameter,missingType.parameter
   */
  public function processDefinition(&$definition, $plugin_id): void {
    parent::processDefinition($definition, $plugin_id);
    // Create local.storage if not available.
    $definition[StringDefinition::STRING_ID] = $definition[StringDefinition::ID];
    $source = $definition[StringDefinition::STRING_ID];
    $context = $definition[StringDefinition::MSG_CONTEXT] ?? '';

    try {
      // Look up the source string and any existing translation.
      $strings = $this->localStorage->getTranslations([
        'source' => $source,
        'context' => $context,
      ]);
      $string = $strings ? reset($strings) : NULL;
      $location = (string) $definition['provider'];
      if (empty($string)) {
        $string = $this
          ->localStorage
          ->createString([
            'source' => $source,
            'context' => $context,
          ])
          ->addLocation('string', $location)
          ->save();
      }
      $definition[StringDefinition::LOCALE_ID] = $string->getId();
      $definition['getLocations'] = $string->getLocations();
    }
    catch (\Exception $e) {
      // If locale tables don't exist yet (e.g., during test setup),
      // we'll set a default locale ID and empty locations.
      $definition[StringDefinition::LOCALE_ID] = 0;
      $definition['getLocations'] = [];
    }

    try {
      new StringDefinition($definition);
    }
    catch (InvalidStringDefinitionException $exception) {
      throw new PluginException($exception->getMessage());
    }
  }

  // phpcs:disable
  /**
   * {@inheritdoc}
   */
  protected function findDefinitions(): array {
    // Doing this to make sure our custom processDefinition() method is called.
    return parent::findDefinitions();
  }
  // phpcs:enable

}
