<?php

namespace Drupal\string\Translator;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\StringTranslation\Translator\StaticTranslation;
use Drupal\string\StringManager;
use Drupal\string\DTO\StringDefinition;

/**
 * Provides translated values for strings.
 *
 * The purpose of this translator is to provide translation that can be
 * shipped with the module.
 *
 * E.g. Consider the following item inside my_module.string.yml,
 * ```
 * # Demo: String with a placeholder.
 * dashboard.welcome_message.short:
 *   default: "Hi @name!"
 *   translations:
 *     en: "Hi @name!"
 *     de: "Moin @name!"
 * ```
 * Then, the "de" string for `dashboard.welcome_message.short` would be
 * `Moin @name!`.
 *
 * The priority of this translator is kept low so that, any overrides provided
 * via po files or interface translation would take precedence.
 */
class StringTranslation extends StaticTranslation {

  /**
   * The string service.
   *
   * @var \Drupal\string\StringManager
   */
  protected $stringManager;

  /**
   * Constructs a StaticTranslation object.
   *
   * @param \Drupal\string\StringManager $stringManager
   *   The string manager service.
   */
  public function __construct(StringManager $stringManager) {
    $this->stringManager = $stringManager;
    $translations = $this->getTranslationsFromDefinitions();
    parent::__construct($translations);
  }

  /**
   * Go through each string plugin items and extract default values.
   *
   * @return array
   *   Array of translations defined in string plugin.
   */
  protected function getTranslationsFromDefinitions() {
    $definitions = $this->stringManager->getDefinitions();
    $translations = [];
    foreach ($definitions as $def) {
      $definition = new StringDefinition($def);
      $items = $definition->getTranslations() ?? [];
      foreach ($items as $lang_code => $value) {
        if ($value) {
          if (is_array($value)) {
            $key = implode(Poitem::DELIMITER, [
              $definition->getId(),
              $definition->getId(),
            ]);
            $value = implode(Poitem::DELIMITER, $value);
          }
          else {
            $key = $definition->getId();
          }
          $context = $definition->getContext();
          $translations[$lang_code][$key][$context] = $value;
          $msg_id = $definition->getDefaultValue();
          if ($msg_id) {
            $translations[$lang_code][$msg_id][$context] = $value;
          }
        }
      }
    }
    return $translations;
  }

  /**
   * {@inheritdoc}
   */
  public function getStringTranslation($langcode, $string, $context) {
    return $this->translations[$langcode][$string][$context] ?? FALSE;
  }

}
