<?php

namespace Drupal\string\Translator;

use Drupal\string\DTO\StringDefinition;
use Drupal\Component\Gettext\PoItem;
use Drupal\Core\StringTranslation\Translator\StaticTranslation;
use Drupal\string\StringManager;

/**
 * Provides default values for strings.
 *
 * The purpose of this translator is to provide default translation when
 * no other method provides any. The default is picked up from the string
 * plugin item defined in my_module.string.yml.
 *
 * E.g. Consider the following item inside my_module.string.yml,
 * ```
 * # Demo: String with a placeholder.
 * dashboard.welcome_message.short:
 *   default: "Hello @name!"
 * ```
 * Then, the default string for `dashboard.welcome_message.short` would be
 * `Hello @name!`.
 *
 * The priority of this translator is kept low so that, any overrides provided
 * via po files or interface translation would take precendence.
 */
class StringTranslationDefaults extends StaticTranslation {

  /**
   * The string service.
   *
   * @var \Drupal\string\StringManager
   */
  protected $stringManager;

  /**
   * Constructs a StaticTranslation object.
   *
   * @param \Drupal\string\StringManager $stringManager
   *   The string manager service.
   */
  public function __construct(StringManager $stringManager) {
    $this->stringManager = $stringManager;
    $translations = $this->getDefaultValues();
    parent::__construct($translations);
  }

  /**
   * Go through each string plugin items and extract default values.
   *
   * @return array
   *   Array of translations defined in string plugin.
   */
  protected function getDefaultValues() {
    $definitions = $this->stringManager->getDefinitions();
    $translations = [];
    foreach ($definitions as $def) {
      $definition = new StringDefinition($def);
      if ($definition->getDefaultValuePlural()) {
        $value = [
          0 => $definition->getDefaultValue(),
          1 => $definition->getDefaultValuePlural(),
        ];
      }
      else {
        $value = $definition->getDefaultValue();
      }
      if ($value) {
        if (is_array($value)) {
          $key = implode(Poitem::DELIMITER, [
            $definition->getStringId(),
            $definition->getStringId(),
          ]);
          $value = implode(Poitem::DELIMITER, $value);
        }
        else {
          $key = $definition->getStringId();
        }
        $context = $definition->getContext() ?? '';
        $translations[$key][$context] = $value;
        $msg_id = $definition->getDefaultValue() ?? NULL;
        if ($msg_id) {
          $translations[$msg_id][$context] = $value;
        }
      }
    }
    return $translations;
  }

  /**
   * {@inheritdoc}
   */
  public function getStringTranslation($langcode, $string, $context) {
    return $this->translations[$string][$context] ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function reset(): void {
    $this->translations = $this->getDefaultValues();
  }

}
