<?php

namespace Drupal\Tests\string\Kernel;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\KernelTests\KernelTestBase;
use Drupal\string\Translator\StringTranslation;

/**
 * Tests the string translation functionality.
 *
 * @group string
 */
class StringTranslationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'string',
    'string_demo_test',
    'locale',
    'language',
  ];

  /**
   * The string manager.
   *
   * @var \Drupal\string\StringManager
   */
  protected $stringManager;

  /**
   * The string translator.
   *
   * @var \Drupal\string\Translator\StringTranslation
   */
  protected $stringTranslator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install all required locale schema tables.
    $this->installSchema('locale', [
      'locales_source',
      'locales_target',
      'locales_location',
    ]);

    // Install language configuration.
    $this->installConfig(['language']);

    // Install string and string_demo_test config.
    $this->installConfig(['string', 'string_demo_test']);

    // Now that the database tables are created,
    // we can safely instantiate the services.
    $this->stringManager = $this->container->get('plugin.manager.string');
    $this->stringTranslator = new StringTranslation($this->stringManager);
  }

  /**
   * Tests basic string translation.
   */
  public function testBasicTranslation(): void {
    $string_id = 'string_demo_test.dashboard.welcome_message.short';
    $context = '';
    $langcode = 'en';

    $translation = $this->stringTranslator->getStringTranslation($langcode, $string_id, $context);
    $this->assertEquals('Hello @name!', $translation);
  }

  /**
   * Tests translation with placeholders.
   */
  public function testTranslationWithPlaceholders(): void {
    $string_id = 'string_demo_test.dashboard.welcome_message.short';
    $context = '';
    $langcode = 'en';
    $args = ['@name' => 'Dave'];

    $translation = $this->stringTranslator->getStringTranslation($langcode, $string_id, $context);
    $translation = strtr($translation, $args);
    $this->assertEquals('Hello Dave!', $translation);
  }

  /**
   * Tests plural translation.
   */
  public function testPluralTranslation(): void {
    $string_id = 'string_demo_test.search.result.items_count';
    $string_id = $string_id . PoItem::DELIMITER . $string_id;
    $value = '@count item found' . PoItem::DELIMITER . '@count items found';
    $context = '';
    $langcode = 'en';
    // Test singular form.
    $translation = $this->stringTranslator->getStringTranslation($langcode, $string_id, $context);
    $this->assertEquals($value, $translation);
  }

  /**
   * Tests translation with context.
   */
  public function testTranslationWithContext(): void {
    $string_id = 'string_demo_test.dashboard.greetings.short';
    $context = 'short_greeting';
    $langcode = 'en';

    $translation = $this->stringTranslator->getStringTranslation($langcode, $string_id, $context);
    $this->assertEquals('Sup @name!', $translation);
  }

  /**
   * Tests non-existent translation.
   */
  public function testNonExistentTranslation(): void {
    $string_id = 'non.existent.string';
    $context = '';
    $langcode = 'en';

    $translation = $this->stringTranslator->getStringTranslation($langcode, $string_id, $context);
    $this->assertFalse($translation);
  }

  /**
   * Provides some test data for formatPlural()
   *
   * @return array
   *   Test data for formatPlural().
   */
  public static function providerTestFormatPlural() {
    return [
      [0, 'string_demo_test.search.result.items_count', [], [], '0 items found'],
      [1, 'string_demo_test.search.result.items_count', [], [], '1 item found'],
      [2, 'string_demo_test.search.result.items_count', [], [], '2 items found'],
    ];
  }

  /**
   * Tests the formatPlural() method.
   *
   * @dataProvider providerTestFormatPlural
   */
  public function testFormatPlural($count, $string_id, array $args, array $options, $expected): void {
    $id = $string_id . PoItem::DELIMITER . $string_id;
    $context = '';
    $langcode = 'en';
    // Test singular form.
    $translation = $this->stringTranslator->getStringTranslation($langcode, $id, $context);
    $markup = PluralTranslatableMarkup::createFromTranslatedString($count, $translation, $args, $options);
    $this->assertEquals($expected, $markup->render());
  }

}
