<?php

namespace Drupal\Tests\string\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\locale\StringInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\locale\StringStorageInterface;
use Drupal\string\StringManager;

/**
 * @coversDefaultClass \Drupal\string\StringManager
 * @group string
 */
class StringManagerTest extends UnitTestCase {

  /**
   * The string manager.
   *
   * @var \Drupal\string\StringManager
   */
  protected $stringManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $themeHandler;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cacheBackend;

  /**
   * The string storage.
   *
   * @var \Drupal\locale\StringStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->themeHandler = $this->createMock(ThemeHandlerInterface::class);
    $this->cacheBackend = $this->createMock(CacheBackendInterface::class);
    $this->stringStorage = $this->createMock(StringStorageInterface::class);

    // Configure the mock to return an empty array for getTranslations().
    $this->stringStorage->expects($this->any())
      ->method('getTranslations')
      ->willReturn([]);

    // Configure the mock to return a valid object for createString().
    $mockString = $this->createMock(StringInterface::class);
    $mockString->expects($this->any())
      ->method('addLocation')
      ->willReturnSelf();
    $mockString->expects($this->any())
      ->method('save')
      ->willReturnSelf();
    $mockString->expects($this->any())
      ->method('getId')
      ->willReturn(1);

    $this->stringStorage->expects($this->any())
      ->method('createString')
      ->willReturn($mockString);

    $namespaces = new \ArrayObject([
      'Drupal\\string' => [__DIR__ . '/../../../../src'],
    ]);

    $this->stringManager = new StringManager(
      $namespaces,
      $this->moduleHandler,
      $this->cacheBackend,
      $this->stringStorage,
      $this->themeHandler
    );
  }

  /**
   * @covers ::checkForNamespace
   */
  public function testCheckForNamespace(): void {
    $reflection = new \ReflectionClass($this->stringManager);
    $method = $reflection->getMethod('checkForNamespace');
    $method->setAccessible(TRUE);

    // Test valid namespace.
    $this->assertNull($method->invoke($this->stringManager, 'valid.namespace'));

    // Test invalid namespace.
    $this->expectException(PluginException::class);
    $method->invoke($this->stringManager, 'invalidnamespace');
  }

  /**
   * @covers ::getDefinitions
   */
  public function testGetDefinitions(): void {
    $this->moduleHandler->expects($this->once())
      ->method('getModuleDirectories')
      ->willReturn(['string' => __DIR__ . '/../../../../']);

    // Explicitly forbid themeHandler call to `getThemeDirectories`.
    // Even though the call is safe to use within Drupal core, it leads
    // to crashes when working with module that does config override,
    // like symfony_mailer and domain_config.
    $this->themeHandler->expects($this->never())
      ->method('getThemeDirectories')
      ->willReturn([]);

    $definitions = $this->stringManager->getDefinitions();
    $this->assertIsArray($definitions);
  }

  /**
   * @covers ::processDefinition
   */
  public function testProcessDefinition(): void {
    $definition = [
      'id' => 'test.string',
      'default' => 'Test String',
      'provider' => 'test_module',
    ];

    $this->stringManager->processDefinition($definition, 'test.string');
    $this->assertEquals('test.string', $definition['id']);
    $this->assertEquals('Test String', $definition['default']);
  }

}
