<?php

namespace Drupal\stripe_api\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\stripe_api\Event\StripeApiWebhookEvent;
use Drupal\stripe_api\StripeApiService;
use Stripe\Event;
use Stripe\Webhook;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class StripeApiWebhook.
 *
 * Provides the route functionality for stripe_api.webhook route.
 */
class StripeApiWebhook extends ControllerBase {

  /**
   * Stripe API service.
   *
   * @var \Drupal\stripe_api\StripeApiService
   */
  protected StripeApiService $stripeApi;

  /**
   * Event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $eventDispatcher;

  public function __construct(StripeApiService $stripe_api, EventDispatcherInterface $event_dispatcher) {
    $this->stripeApi = $stripe_api;
    $this->eventDispatcher = $event_dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('stripe_api.stripe_api'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * Captures the incoming webhook request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A Response object.
   */
  public function handleIncomingWebhook(Request $request): Response {
    $config = $this->config('stripe_api.settings');
    if ($config->get('enable_webhooks') === FALSE) {
      return new Response('Incoming webhooks are disabled by the Stripe API module configuration.', Response::HTTP_FORBIDDEN);
    }

    $content = $request->getContent();

    if (!$event = $this->getEventFromRequest($request)) {
      $this->getLogger('stripe_api')
        ->error('Invalid webhook event: @body', [
          '@body' => $content,
        ]);
      return new Response(NULL, Response::HTTP_FORBIDDEN);
    }

    if ($config->get('log_webhooks')) {
      /** @var \Drupal\Core\Logger\LoggerChannelInterface $logger */
      $logger = $this->getLogger('stripe_api');
      $logger->info("Stripe webhook received event:\n @event", ['@event' => (string) $event]);
    }

    // Dispatch the webhook event.
    try {
      $webhook_event = new StripeApiWebhookEvent($event->type, $event);
      $this->eventDispatcher->dispatch($webhook_event, 'stripe_api.webhook');

      return new Response('Okay', Response::HTTP_OK);
    }
    catch (\RuntimeException $e) {
      // Handle temporary failures (e.g., lock contention) - request retry
      $this->getLogger('stripe_api')
        ->warning('Webhook processing temporarily unavailable: @error', [
          '@error' => $e->getMessage(),
        ]);

      // Return 503 Service Unavailable to trigger Stripe retry
      return new Response(
        $e->getMessage(),
        Response::HTTP_SERVICE_UNAVAILABLE,
        ['Retry-After' => '5']
      );
    }
  }

  /**
   * Determines if a webhook is valid.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return bool|\Stripe\Event
   *   Returns a Stripe Event object or false if validation fails.
   */
  private function getEventFromRequest(Request $request): ?Event {
    $secret = $this->stripeApi->getWebhookSigningSecret();
    try {
      $event = Webhook::constructEvent(
        $request->getContent(), $request->headers->get('stripe-signature'), $secret
      );
    }
    catch (\UnexpectedValueException $e) {
      $this->getLogger('stripe_api')
        ->error('Invalid payload: @body', [
          '@body' => $e->getMessage(),
        ]);
      return NULL;
    }
    catch (\Exception $e) {
      $this->getLogger('stripe_api')
        ->error('Invalid signature: @body', [
          '@body' => $e->getMessage(),
        ]);

      return NULL;
    }

    return $event;
  }

}
