<?php

declare(strict_types=1);

namespace Drupal\stripe_sync\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Stripe\StripeClient;
use Stripe\Exception\ApiErrorException;
use Drupal\user\UserInterface;

final class CheckoutController extends ControllerBase {

  /**
   * Starts a Stripe Checkout Session for a Product or Price.
   * Accepts either a product ID (prod_...) or price ID (price_...).
   * - Recurring price  -> mode=subscription
   * - One-time price   -> mode=payment
   */
  public function start(?string $id = null) {

    // If not logged in, send to login and come back here after.
    if ($this->currentUser()->isAnonymous()) {
      $destination = \Drupal::request()->getRequestUri(); // keeps query string too
      $this->messenger()->addStatus($this->t('Please log in or create an account to continue to checkout.'));
      return $this->redirect('user.login', [], ['query' => ['destination' => $destination]]);
      // If you prefer to send them to register instead:
      // return $this->redirect('user.register', [], ['query' => ['destination' => $destination]]);
    }

    if (!$this->currentUser()->isAuthenticated()) {
      throw new AccessDeniedHttpException('Login required to start checkout.');
    }

    if (!$id) {
      $this->messenger()->addError($this->t('Please provide a Stripe Product ID (prod_…) or Price ID (price_…).'));
      return $this->redirect('<front>');
    }

    $secret = $this->resolveStripeSecretFromConfig();
    if ($secret === '') {
      $this->messenger()->addError($this->t('Stripe secret key is not configured. Set it at /admin/config/system/stripe.'));
      return $this->redirect('<front>');
    }

    // Module settings.
    $cfgMod = $this->config('stripe_sync.settings');
    $allowPayment = (bool) ($cfgMod->get('allow_payment_mode') ?? TRUE);
    $requireAccessDays = (bool) ($cfgMod->get('require_access_days_for_payment') ?? FALSE);
    $enableInvoiceForPayment = (bool) ($cfgMod->get('enable_invoice_creation_for_payment') ?? FALSE);
    $dupPolicy = (string) ($cfgMod->get('prevent_duplicate_subscriptions') ?? 'same_product'); // none|same_product|any

    $stripe = new StripeClient($secret);

    // 1) Resolve a Price ID and whether it is recurring.
    $priceId = null;
    $isRecurring = false;
    $price = null;

    try {
      if (str_starts_with($id, 'prod_')) {
        // Expand default_price so we know which price we're using.
        $product = $stripe->products->retrieve($id, ['expand' => ['default_price']]);
        $priceId = $product->default_price?->id ?? null;
        if (!$priceId) {
          $this->messenger()->addError($this->t('This product (%id) has no default price. Set a default price in Stripe and try again.', ['%id' => $id]));
          return $this->redirect('<front>');
        }
        // Retrieve price and expand product for metadata fallback.
        $price = $stripe->prices->retrieve($priceId, ['expand' => ['product']]);
      }
      elseif (str_starts_with($id, 'price_')) {
        $priceId = $id;
        // Retrieve price and expand product for metadata fallback.
        $price = $stripe->prices->retrieve($priceId, ['expand' => ['product']]);
      }
      else {
        $this->messenger()->addError($this->t('Please pass a Stripe Price ID (price_…) or Product ID (prod_…). Given: %id', ['%id' => $id]));
        return $this->redirect('<front>');
      }

      $isRecurring = (($price->type ?? null) === 'recurring') || !empty($price->recurring);
    }
    catch (ApiErrorException $e) {
      $this->messenger()->addError($this->t('Stripe error while checking the item: @msg', ['@msg' => $e->getMessage()]));
      return $this->redirect('<front>');
    }

    // 2) Build customer context (read field names from our module settings).
    $field_customer = (string) ($cfgMod->get('field_customer_id') ?? 'field_stripe_customer_id');
    $field_sub_id   = (string) ($cfgMod->get('field_subscription_id') ?? 'field_stripe_subscription_id');

    $uid = (int) $this->currentUser()->id();
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->entityTypeManager()->getStorage('user')->load($uid);
    $email = $user->getEmail() ?: '';
    $customerId = $user->hasField($field_customer) ? (string) ($user->get($field_customer)->value ?? '') : '';
    $localSubId = ($field_sub_id && $user->hasField($field_sub_id))
      ? trim((string) ($user->get($field_sub_id)->value ?? ''))
      : '';

    // 2a) Duplicate-subscription guard #1: check user's locally stored subscription first.
    if ($isRecurring && $dupPolicy !== 'none' && $localSubId !== '') {
      try {
        $existing = $stripe->subscriptions->retrieve($localSubId, [
          'expand' => ['items.data.price.product'],
        ]);

        $status = strtolower((string) ($existing->status ?? ''));
        $isTerminal = in_array($status, ['canceled', 'incomplete_expired'], true);
        if (!$isTerminal) {
          $block = false;

          if ($dupPolicy === 'any') {
            $block = true;
          }
          elseif ($dupPolicy === 'same_product') {
            $targetProductId = null;
            $targetProductName = '';
            if (!empty($price->product)) {
              $targetProductId = is_string($price->product) ? $price->product : ($price->product->id ?? null);
              $targetProductName = is_object($price->product) ? ((string) ($price->product->name ?? '')) : '';
            }
            if ($targetProductId && !empty($existing->items?->data) && is_array($existing->items->data)) {
              foreach ($existing->items->data as $item) {
                $p = $item->price->product ?? null;
                $pid = is_string($p) ? $p : ($p->id ?? null);
                if ($pid && $pid === $targetProductId) {
                  $block = true;
                  break;
                }
              }
            }
          }

          if ($block) {
            $this->denyNewSubscription($targetProductName ?? '', $status, $uid);
            return $this->redirect('entity.user.canonical', ['user' => $uid]);
          }
        }
      }
      catch (\Throwable $e) {
        $this->getLogger('stripe_sync')->warning(
          'Local subscription guard failed for user @uid (sub @sub): @msg',
          ['@uid' => $uid, '@sub' => $localSubId, '@msg' => $e->getMessage()]
        );
      }
    }

    // 2b) Duplicate-subscription guard #2: check by Stripe customer(s).
    // Collect all candidate customers to check (stored ID + any by email).
    $candidateCustomerIds = [];
    if ($customerId !== '') {
      $candidateCustomerIds[] = $customerId;
    }

    // Also look up by email to catch duplicate Stripe customers from past tests.
    if ($email !== '') {
      try {
        $found = $stripe->customers->all(['email' => $email, 'limit' => 10]);
        foreach ($found->data as $c) {
          if (!in_array($c->id, $candidateCustomerIds, true)) {
            $candidateCustomerIds[] = (string) $c->id;
          }
        }
        // If no stored customer but we found one, persist the first one for next time.
        if ($customerId === '' && !empty($candidateCustomerIds)) {
          $customerId = $candidateCustomerIds[0];
          if ($user->hasField($field_customer)) {
            $user->set($field_customer, $customerId);
            $user->save();
          }
        }
      }
      catch (\Throwable $e) {
        // Non-fatal; continue without customer id.
        $this->getLogger('stripe_sync')->warning('Customer email lookup failed for @mail: @msg', [
          '@mail' => $email,
          '@msg'  => $e->getMessage(),
        ]);
      }
    }

    if ($isRecurring && !empty($candidateCustomerIds) && $dupPolicy !== 'none') {
      $targetProductId = null;
      $targetProductName = '';
      if (!empty($price->product)) {
        $targetProductId = is_string($price->product) ? $price->product : ($price->product->id ?? null);
        $targetProductName = is_object($price->product) ? ((string) ($price->product->name ?? '')) : '';
      }

      try {
        $terminal = ['canceled', 'incomplete_expired'];
        foreach ($candidateCustomerIds as $cidToCheck) {
          $subs = $stripe->subscriptions->all([
            'customer' => $cidToCheck,
            'status' => 'all',
            'limit' => 20,
            'expand' => ['data.items.data.price.product'],
          ]);

          foreach ($subs->data as $sub) {
            $status = strtolower((string) ($sub->status ?? ''));
            if (in_array($status, $terminal, true)) {
              continue; // finished; ignore
            }

            if ($dupPolicy === 'any') {
              $this->denyNewSubscription($targetProductName, $status, $uid);
              return $this->redirect('entity.user.canonical', ['user' => $uid]);
            }

            if ($dupPolicy === 'same_product' && $targetProductId) {
              $match = false;
              if (!empty($sub->items?->data) && is_array($sub->items->data)) {
                foreach ($sub->items->data as $item) {
                  $p = $item->price->product ?? null;
                  $pid = is_string($p) ? $p : ($p->id ?? null);
                  if ($pid && $pid === $targetProductId) {
                    $match = true;
                    break;
                  }
                }
              }
              if ($match) {
                $this->denyNewSubscription($targetProductName, $status, $uid);
                return $this->redirect('entity.user.canonical', ['user' => $uid]);
              }
            }
          }
        }
      }
      catch (\Throwable $e) {
        // Fail open: allow checkout if the check itself fails.
        $this->getLogger('stripe_sync')->warning('Duplicate-subscription check failed for user @uid: @msg', [
          '@uid' => $uid,
          '@msg' => $e->getMessage(),
        ]);
      }
    }

    // Success/cancel URLs.
    $request = \Drupal::requestStack()->getCurrentRequest();
    $base = $request->getSchemeAndHttpHost();
    $successUrl = $base . '/checkout/success?session_id={CHECKOUT_SESSION_ID}';
    $cancelUrl  = $base . '/checkout/cancel';

    // Common params for session.create.
    $params = [
      'line_items' => [[ 'price' => $priceId, 'quantity' => 1 ]],
      'success_url' => $successUrl,
      'cancel_url'  => $cancelUrl,
      'client_reference_id' => (string) $uid,
      'allow_promotion_codes' => true,
    ];

    if ($customerId !== '') {
      $params['customer'] = $customerId;
    } else {
      if ($isRecurring) {
        // SUBSCRIPTION: don't use customer_creation here.
        if ($email !== '') {
          $params['customer_email'] = $email;
        }
      } else {
        // PAYMENT (one-time): ok to auto-create a Customer.
        $params['customer_creation'] = 'always';
        if ($email !== '') {
          $params['customer_email'] = $email;
        }
      }
    }
    

    // Mode-specific behavior.
    if ($isRecurring) {
      // SUBSCRIPTION.
      $params['mode'] = 'subscription';
      $params['subscription_data'] = [
        'metadata' => ['drupal_uid' => (string) $uid],
      ];
    }
    else {
      // ONE-TIME PAYMENT.
      if (!$allowPayment) {
        $this->messenger()->addError($this->t('This item is a one-time price. One-time purchases are currently disabled.'));
        return $this->redirect('<front>');
      }

      // Prefer price metadata; fallback to product metadata.
      $accessDays = 0;
      if (!empty($price->metadata) && isset($price->metadata['access_days'])) {
        $accessDays = (int) $price->metadata['access_days'];
      }
      elseif (!empty($price->product?->metadata) && isset($price->product->metadata['access_days'])) {
        $accessDays = (int) $price->product->metadata['access_days'];
      }

      if ($requireAccessDays && $accessDays <= 0) {
        $this->messenger()->addError($this->t('This one-time price has no access duration configured. Please contact support.'));
        return $this->redirect('<front>');
      }

      $params['mode'] = 'payment';

      // Copy drupal_uid (and access_days if any) into the PaymentIntent metadata.
      $pidMeta = ['drupal_uid' => (string) $uid];
      if ($accessDays > 0) {
        $pidMeta['access_days'] = (string) $accessDays;
      }
      $params['payment_intent_data'] = ['metadata' => $pidMeta];

      // Optionally create an invoice for the one-time payment.
      if ($enableInvoiceForPayment) {
        $params['invoice_creation'] = ['enabled' => true];
      }
    }

    // 4) Create checkout session and redirect to Stripe.
    try {
      $session = $stripe->checkout->sessions->create($params);
      return new TrustedRedirectResponse($session->url, 303);
    }
    catch (ApiErrorException $e) {
      $this->messenger()->addError($this->t('Stripe error while creating Checkout: @msg', ['@msg' => $e->getMessage()]));
      return $this->redirect('<front>');
    }
  }

  /**
   * Helper: add a friendly message when we block duplicate subscriptions.
   */
  private function denyNewSubscription(string $productName, string $status, int $uid): void {
    $portalLink = Link::fromTextAndUrl(
      $this->t('Open billing portal'),
      Url::fromRoute('stripe_sync.portal_self')
    )->toString();

    $label = $productName !== '' ? $productName : $this->t('this product');
    $s = strtolower($status);

    if (in_array($s, ['active', 'trialing'], true)) {
      $this->messenger()->addWarning(
        $this->t('You already have an @status subscription for @product. @portal', [
          '@status' => $s,
          '@product' => $label,
          '@portal' => $portalLink,
        ])
      );
    }
    elseif (in_array($s, ['past_due', 'unpaid', 'incomplete', 'paused'], true)) {
      $this->messenger()->addWarning(
        $this->t('Your subscription for @product is @status. Please resolve it in the billing portal. @portal', [
          '@status' => $s,
          '@product' => $label,
          '@portal' => $portalLink,
        ])
      );
    }
    else {
      $this->messenger()->addWarning(
        $this->t('You already have a subscription for @product (status: @status). @portal', [
          '@product' => $label,
          '@status' => $s,
          '@portal' => $portalLink,
        ])
      );
    }
  }

  /**
   * Success page at /checkout/success?session_id=cs_...
   */
  public function success(): array {
    $sessionId = (string) (\Drupal::request()->query->get('session_id') ?? '');
    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['stripe-checkout-success']],
      'title' => ['#markup' => '<h1>' . $this->t('Thanks! Your payment was successful.') . '</h1>'],
    ];

    if ($sessionId === '') {
      $this->messenger()->addWarning($this->t('Missing session_id. Showing a generic confirmation.'));
      $build['body'] = ['#markup' => '<p>' . $this->t('You can now access your content.') . '</p>'];
      return $build;
    }

    $secret = $this->resolveStripeSecretFromConfig();
    if ($secret === '') {
      $this->messenger()->addWarning($this->t('Stripe config not available to show details.'));
      $build['body'] = ['#markup' => '<p>' . $this->t('Your payment completed. You can now access your content.') . '</p>'];
      return $build;
    }

    try {
      $stripe = new StripeClient($secret);
      $session = $stripe->checkout->sessions->retrieve($sessionId, [
        'expand' => ['payment_intent', 'subscription', 'customer'],
      ]);

      $mode = $session->mode;
      $amount = $session->amount_total ? number_format($session->amount_total / 100, 2) : null;
      $currency = strtoupper((string) ($session->currency ?? ''));
      $subId = is_string($session->subscription ?? null) ? $session->subscription : ($session->subscription->id ?? '');
      $cust = is_string($session->customer ?? null) ? $session->customer : ($session->customer->id ?? '');

      $lines = [];
      $lines[] = $this->t('Mode: @mode', ['@mode' => $mode]);
      if ($amount && $currency) {
        $lines[] = $this->t('Amount: @amt @cur', ['@amt' => $amount, '@cur' => $currency]);
      }
      if ($cust) {
        $lines[] = $this->t('Customer: @cid', ['@cid' => $cust]);
      }
      if ($subId) {
        $lines[] = $this->t('Subscription: @sid', ['@sid' => $subId]);
      }

      $build['body'] = [
        '#theme' => 'item_list',
        '#title' => $this->t('Checkout details'),
        '#items' => $lines,
      ];
      $build['next'] = ['#markup' => '<p>' . $this->t('You can now access your content. If you don’t see access yet, refresh once.') . '</p>'];
    }
    catch (ApiErrorException $e) {
      $this->messenger()->addWarning($this->t('Could not load Checkout session: @msg', ['@msg' => $e->getMessage()]));
      $build['body'] = ['#markup' => '<p>' . $this->t('Your payment completed. You can now access your content.') . '</p>'];
    }

    return $build;
  }

  /**
   * Cancel page at /checkout/cancel
   */
  public function cancel(): array {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['stripe-checkout-cancel']],
      'title' => ['#markup' => '<h1>' . $this->t('Checkout canceled') . '</h1>'],
      'body' => ['#markup' => '<p>' . $this->t('No payment was made. You can try again anytime.') . '</p>'],
    ];
  }

  /**
   * Create a Stripe Billing Portal session for the current user and redirect.
   * Route: /billing
   */
  public function portalSelf() {
    if (!$this->currentUser()->isAuthenticated()) {
      throw new AccessDeniedHttpException('Login required.');
    }

    $secret = $this->resolveStripeSecretFromConfig();
    if ($secret === '') {
      $this->messenger()->addError($this->t('Stripe secret key is not configured.'));
      return $this->redirect('<front>');
    }

    $cfg = $this->config('stripe_sync.settings');
    $field_customer = (string) ($cfg->get('field_customer_id') ?? 'field_stripe_customer_id');

    $uid = (int) $this->currentUser()->id();
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->entityTypeManager()->getStorage('user')->load($uid);

    $customerId = $user && $user->hasField($field_customer)
      ? (string) ($user->get($field_customer)->value ?? '')
      : '';

    if ($customerId === '') {
      $this->messenger()->addError($this->t('No Stripe customer is linked to your account yet.'));
      return $this->redirect('entity.user.canonical', ['user' => $uid]);
    }

    $request = \Drupal::requestStack()->getCurrentRequest();
    $base = $request->getSchemeAndHttpHost();
    $returnUrl = $base . '/user';

    try {
      $stripe = new StripeClient($secret);
      $session = $stripe->billingPortal->sessions->create([
        'customer' => $customerId,
        'return_url' => $returnUrl,
      ]);
      return new TrustedRedirectResponse($session->url, 303);
    }
    catch (ApiErrorException $e) {
      // Common cause: Customer portal not enabled in Stripe Dashboard (Test).
      $this->messenger()->addError($this->t('Could not open billing portal: @msg', ['@msg' => $e->getMessage()]));
      return $this->redirect('entity.user.canonical', ['user' => $uid]);
    }
  }

  /**
   * Admin: open a user's billing portal (for support).
   * Route: /admin/people/{user}/billing
   */
  public function portalForUser(UserInterface $user) {
    $secret = $this->resolveStripeSecretFromConfig();
    if ($secret === '') {
      $this->messenger()->addError($this->t('Stripe secret key is not configured.'));
      return $this->redirect('entity.user.canonical', ['user' => $user->id()]);
    }

    $cfg = $this->config('stripe_sync.settings');
    $field_customer = (string) ($cfg->get('field_customer_id') ?? 'field_stripe_customer_id');

    $customerId = $user->hasField($field_customer) ? (string) ($user->get($field_customer)->value ?? '') : '';
    if ($customerId === '') {
      $this->messenger()->addError($this->t('This user is not linked to a Stripe customer.'));
      return $this->redirect('entity.user.canonical', ['user' => $user->id()]);
    }

    $request = \Drupal::requestStack()->getCurrentRequest();
    $base = $request->getSchemeAndHttpHost();
    $returnUrl = $base . '/user/' . $user->id();

    try {
      $stripe = new StripeClient($secret);
      $session = $stripe->billingPortal->sessions->create([
        'customer' => $customerId,
        'return_url' => $returnUrl,
      ]);
      return new TrustedRedirectResponse($session->url, 303);
    }
    catch (ApiErrorException $e) {
      $this->messenger()->addError($this->t('Could not open billing portal: @msg', ['@msg' => $e->getMessage()]));
      return $this->redirect('entity.user.canonical', ['user' => $user->id()]);
    }
  }

  /**
   * Reads secret from stripe.settings (your module’s form writes these keys).
   * Falls back to STRIPE_SECRET_KEY env var if config is empty.
   */
  private function resolveStripeSecretFromConfig(): string {
    $cfg = $this->config('stripe.settings');
    $env = (string) ($cfg->get('environment') ?? 'test'); // 'test' or 'live'
    $keyPath = $env === 'live' ? 'apikey.live.secret' : 'apikey.test.secret';
    $secret = (string) ($cfg->get($keyPath) ?? '');
    if ($secret === '' && getenv('STRIPE_SECRET_KEY')) {
      $secret = (string) getenv('STRIPE_SECRET_KEY');
    }
    return $secret;
  }
}
