<?php

declare(strict_types=1);

namespace Drupal\stripe_sync\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\stripe\Event\StripeEvents;
use Drupal\stripe\Event\StripeWebhookEvent;
use Drupal\user\UserInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Reacts to Stripe webhooks and syncs user roles/fields.
 */
final class StripeWebhookSubscriber implements EventSubscriberInterface {

  // Default role machine names (used if config is empty).
  private const ROLE_ACTIVE_DEFAULT   = 'stripe_member';
  private const ROLE_PAST_DUE_DEFAULT = 'stripe_member_past_due';
  private const ROLE_INACTIVE_DEFAULT = 'stripe_member_inactive';

  // Default user field machine names (used if config is empty).
  private const FIELD_CUSTOMER_ID_DEFAULT     = 'field_stripe_customer_id';      // REQUIRED
  private const FIELD_SUBSCRIPTION_ID_DEFAULT = 'field_stripe_subscription_id';  // optional
  private const FIELD_SUB_STATUS_DEFAULT      = 'field_subscription_status';     // optional
  private const FIELD_SUB_EXPIRES_DEFAULT     = 'field_subscription_expires';    // optional (Timestamp)
  private const FIELD_CHECKOUT_MODE_DEFAULT   = 'field_checkout_mode';           // optional (List text / plain text)

  public function __construct(
    private readonly EntityTypeManagerInterface $etm,
    private readonly LoggerChannelInterface $logger,
    private readonly ConfigFactoryInterface $configFactory,
  ) {}

  public static function getSubscribedEvents(): array {
    return [StripeEvents::WEBHOOK => 'onStripeWebhook'];
  }

  /**
   * Convenience getters that read from config with fallbacks.
   */
  private function cfg(string $key, ?string $fallback = null): string {
    $val = (string) ($this->configFactory->get('stripe_sync.settings')->get($key) ?? '');
    return $val !== '' ? $val : (string) $fallback;
  }
  private function roleActive(): string   { return $this->cfg('role_active',   self::ROLE_ACTIVE_DEFAULT); }
  private function rolePastDue(): string  { return $this->cfg('role_past_due', self::ROLE_PAST_DUE_DEFAULT); }
  private function roleInactive(): string { return $this->cfg('role_inactive', self::ROLE_INACTIVE_DEFAULT); }

  private function fieldCustomerId(): string { return $this->cfg('field_customer_id', self::FIELD_CUSTOMER_ID_DEFAULT); }
  private function fieldSubId(): string      { return $this->cfg('field_subscription_id', self::FIELD_SUBSCRIPTION_ID_DEFAULT); }
  private function fieldSubStatus(): string  { return $this->cfg('field_subscription_status', self::FIELD_SUB_STATUS_DEFAULT); }
  private function fieldSubExpires(): string { return $this->cfg('field_subscription_expires', self::FIELD_SUB_EXPIRES_DEFAULT); }
  private function fieldCheckoutMode(): string { return $this->cfg('field_checkout_mode', self::FIELD_CHECKOUT_MODE_DEFAULT); }

  /**
   * Main entry point for all Stripe webhook events.
   */
  public function onStripeWebhook(StripeWebhookEvent $event): void {
    $stripeEvent = $event->getEvent(); // \Stripe\Event
    $type = $stripeEvent->type ?? '';
    $obj  = $stripeEvent->data->object ?? null;

    $this->logger->info('Stripe webhook received: @type', ['@type' => $type]);

    // Try mapping to a user via known customer id.
    $customerId = $this->extractCustomerId($type, $obj);
    $user = $customerId ? $this->loadUserByStripeCustomerId($customerId) : NULL;

    // Auto-link if not found yet.
    if (!$user) {
      $user = $this->maybeAutolinkUser($type, $obj, $customerId);
      if ($user) {
        $this->logger->notice('Auto-linked Stripe customer @cid to Drupal user @uid.', [
          '@cid' => $customerId ?? '(none)',
          '@uid' => $user->id(),
        ]);
      }
      else {
        $this->logger->warning('No Drupal user mapped to Stripe customer @cid', ['@cid' => $customerId ?? '(none)']);
        return;
      }
    }

    switch ($type) {
      // ── Subscriptions ──────────────────────────────────────────────────────────
      case 'customer.subscription.created':
      case 'customer.subscription.updated':
      case 'customer.subscription.resumed':
      case 'customer.subscription.paused':
      case 'customer.subscription.pending_update_applied':
      case 'customer.subscription.pending_update_expired':
      case 'customer.subscription.trial_will_end': {
        $status = (string) ($obj->status ?? '');
        $subscriptionId = is_string($obj->id ?? null) ? $obj->id : null;
        $currentPeriodEnd = isset($obj->current_period_end) ? (int) $obj->current_period_end : null;

        $this->persistOptionalFields($user, $status, $subscriptionId, $currentPeriodEnd);
        $this->persistCheckoutMode($user, 'subscription');
        $this->applyStatusRoles($user, $status);
        break;
      }

      case 'customer.subscription.deleted': {
        $this->setRoles($user, [$this->roleInactive()], [$this->roleActive(), $this->rolePastDue()]);
        $status = (string) ($obj->status ?? 'canceled');
        $subscriptionId = is_string($obj->id ?? null) ? $obj->id : null;
        $currentPeriodEnd = isset($obj->current_period_end) ? (int) $obj->current_period_end : null;
        $this->persistOptionalFields($user, $status, $subscriptionId, $currentPeriodEnd);
        $this->persistCheckoutMode($user, 'subscription');
        break;
      }

      // ── Invoices (subscription payments or one-time invoice_creation) ─────────
      case 'invoice.payment_succeeded':
      case 'invoice.paid':
      case 'invoice_payment.paid': {
        // If this invoice belongs to a subscription, treat as subscription.
        if (!empty($obj->subscription) && is_string($obj->subscription)) {
          $this->setRoles($user, [$this->roleActive()], [$this->rolePastDue(), $this->roleInactive()]);
          $this->persistOptionalFields($user, null, $obj->subscription, null);
          $this->persistCheckoutMode($user, 'subscription');
        }
        else {
          // No subscription → likely a one-time invoice (e.g. Checkout invoice_creation).
          $this->setRoles($user, [$this->roleActive()], [$this->roleInactive()]);
          $this->persistCheckoutMode($user, 'payment');
          // Expiry for one-time is handled at payment_intent.succeeded via access_days.
        }
        break;
      }

      case 'invoice.payment_failed': {
        // If tied to a subscription → Past due; otherwise leave Active alone (one-time failure).
        if (!empty($obj->subscription) && is_string($obj->subscription)) {
          $past = $this->rolePastDue();
          $this->setRoles($user, $past ? [$past] : [], []);
          $this->persistCheckoutMode($user, 'subscription');
        }
        else {
          // One-time invoice failed — do not auto-grant access.
          $this->setRoles($user, [$this->roleInactive()], []);
          $this->persistCheckoutMode($user, 'payment');
        }
        break;
      }

      // ── One-time payments (Checkout mode=payment) ─────────────────────────────
      case 'payment_intent.succeeded': {
        // Grant access on successful one-time purchase.
        $this->setRoles($user, [$this->roleActive()], [$this->roleInactive()]);

        // Optional expiry from metadata (set by controller from Price/Product).
        $days = 0;
        if (!empty($obj->metadata?->access_days)) {
          $days = (int) $obj->metadata->access_days;
        }
        if ($days > 0) {
          $expires = \Drupal::time()->getRequestTime() + ($days * 86400);
          $fExpires = $this->fieldSubExpires();
          if ($fExpires && $user->hasField($fExpires)) {
            $user->set($fExpires, $expires);
            $user->save();
          }
        }

        $this->persistCheckoutMode($user, 'payment');
        break;
      }

      // ── Checkout Session completed (either payment or subscription) ───────────
      case 'checkout.session.completed': {
        $subscriptionId = is_string($obj->subscription ?? null) ? $obj->subscription : null;
        $this->persistOptionalFields($user, null, $subscriptionId, null);

        // Persist the actual mode from the Checkout session if present.
        $mode = is_string($obj->mode ?? null) ? $obj->mode : null; // 'payment' | 'subscription' | 'setup'
        if ($mode) {
          $this->persistCheckoutMode($user, $mode);
        }

        $this->logger->info('Checkout session completed for user @uid', ['@uid' => $user->id()]);
        break;
      }

      // ── Setup only (rare in your flow, but supported) ─────────────────────────
      case 'setup_intent.succeeded': {
        $this->persistCheckoutMode($user, 'setup');
        break;
      }

      default:
        // Ignore others or extend as needed.
        break;
    }
  }

  /**
   * Extracts a Stripe customer id from common event objects.
   */
  private function extractCustomerId(string $type, mixed $obj): ?string {
    if (isset($obj->customer) && is_string($obj->customer)) {
      return $obj->customer;
    }
    if ($type === 'checkout.session.completed') {
      if (!empty($obj->customer) && is_string($obj->customer)) {
        return $obj->customer;
      }
    }
    if ($type === 'customer.created' && !empty($obj->id) && is_string($obj->id) && str_starts_with($obj->id, 'cus_')) {
      return $obj->id;
    }
    return null;
  }

  /**
   * Auto-link user if not mapped yet:
   * - Prefer client_reference_id or metadata.drupal_uid on Checkout Session
   * - Else fall back to a unique email match from session/invoice/customer payloads
   * - PaymentIntent email fallback for one-time payments
   * - If found and we know customerId, save field_stripe_customer_id if empty
   */
  private function maybeAutolinkUser(string $type, mixed $obj, ?string $customerId): ?UserInterface {
    $user = null;

    // Checkout (best: explicit UID).
    if ($type === 'checkout.session.completed') {
      if (!empty($obj->client_reference_id) && ctype_digit((string) $obj->client_reference_id)) {
        $user = $this->loadUserByUid((int) $obj->client_reference_id);
      }
      if (!$user && !empty($obj->metadata?->drupal_uid) && ctype_digit((string) $obj->metadata->drupal_uid)) {
        $user = $this->loadUserByUid((int) $obj->metadata->drupal_uid);
      }
      if (!$user && !empty($obj->customer_details?->email) && is_string($obj->customer_details->email)) {
        $user = $this->loadSingleUserByEmail($obj->customer_details->email);
      }
    }

    // Invoice email fallback.
    if (!$user && in_array($type, [
      'invoice.created','invoice.finalized','invoice.payment_succeeded','invoice.payment_failed',
      'invoice.paid','invoice.updated','invoice_payment.paid',
    ], true)) {
      if (!empty($obj->customer_email) && is_string($obj->customer_email)) {
        $user = $this->loadSingleUserByEmail($obj->customer_email);
      }
    }

    // Customer email fallback.
    if (!$user && in_array($type, ['customer.created','customer.updated'], true)) {
      if (!empty($obj->email) && is_string($obj->email)) {
        $user = $this->loadSingleUserByEmail($obj->email);
      }
    }

    // One-time payments: PaymentIntent email fallback.
    if (!$user && $type === 'payment_intent.succeeded') {
      if (!empty($obj->receipt_email) && is_string($obj->receipt_email)) {
        $user = $this->loadSingleUserByEmail($obj->receipt_email);
      }
      if (!$user && !empty($obj->charges?->data) && is_array($obj->charges->data)) {
        foreach ($obj->charges->data as $charge) {
          $email = $charge->billing_details->email ?? null;
          if (is_string($email) && $email !== '') {
            $user = $this->loadSingleUserByEmail($email);
            if ($user) { break; }
          }
        }
      }
    }

    // Save mapping if we found a user and know the customer id.
    $fieldCustomer = $this->fieldCustomerId();
    if ($user && $customerId && $fieldCustomer && $user->hasField($fieldCustomer)) {
      $existing = (string) ($user->get($fieldCustomer)->value ?? '');
      if ($existing === '') {
        $user->set($fieldCustomer, $customerId);
        $user->save();
        $this->logger->info('Saved @field for user @uid = @cid', [
          '@field' => $fieldCustomer,
          '@uid'   => $user->id(),
          '@cid'   => $customerId,
        ]);
      }
    }

    return $user;
  }

  private function loadUserByStripeCustomerId(string $customerId): ?UserInterface {
    $storage = $this->etm->getStorage('user');
    $matches = $storage->loadByProperties([$this->fieldCustomerId() => $customerId]);
    return $matches ? reset($matches) : null;
  }

  private function loadUserByUid(int $uid): ?UserInterface {
    return $this->etm->getStorage('user')->load($uid);
  }

  private function loadSingleUserByEmail(string $email): ?UserInterface {
    $storage = $this->etm->getStorage('user');
    $matches = $storage->loadByProperties(['mail' => mb_strtolower($email)]);
    if (count($matches) === 1) {
      return reset($matches);
    }
    if (count($matches) > 1) {
      $this->logger->warning('Multiple users share email @mail; cannot auto-link.', ['@mail' => $email]);
    }
    return null;
  }

  private function applyStatusRoles(UserInterface $user, string $status): void {
    $s = strtolower($status);

    if (in_array($s, ['active', 'trialing'], true)) {
      $this->setRoles($user, [$this->roleActive()], [$this->rolePastDue(), $this->roleInactive()]);
      return;
    }

    if (in_array($s, ['past_due', 'unpaid'], true)) {
      $past = $this->rolePastDue();
      $this->setRoles($user, $past ? [$past] : [], [$this->roleInactive()]);
      return;
    }

    // Treat incomplete and terminal-ish states as inactive.
    if (in_array($s, ['incomplete', 'canceled', 'incomplete_expired', 'paused'], true) || $s === '') {
      $this->setRoles($user, [$this->roleInactive()], [$this->roleActive(), $this->rolePastDue()]);
    }
  }

  private function setRoles(UserInterface $user, array $add, array $remove): void {
    $changed = false;
    foreach ($add as $rid) {
      if ($rid && !$user->hasRole($rid)) { $user->addRole($rid); $changed = true; }
    }
    foreach ($remove as $rid) {
      if ($rid && $user->hasRole($rid)) { $user->removeRole($rid); $changed = true; }
    }
    if ($changed) {
      $user->save();
      $this->logger->notice('Updated roles for user @uid', ['@uid' => $user->id()]);
    }
  }

  private function persistOptionalFields(
    UserInterface $user,
    ?string $status,
    ?string $subscriptionId,
    ?int $currentPeriodEnd
  ): void {
    $touched = false;

    $fStatus  = $this->fieldSubStatus();
    $fSubId   = $this->fieldSubId();
    $fExpires = $this->fieldSubExpires();

    if ($status !== null && $fStatus && $user->hasField($fStatus)) {
      $user->set($fStatus, $status); $touched = true;
    }
    if ($subscriptionId !== null && $fSubId && $user->hasField($fSubId)) {
      $user->set($fSubId, $subscriptionId); $touched = true;
    }
    if ($currentPeriodEnd !== null && $fExpires && $user->hasField($fExpires)) {
      $user->set($fExpires, $currentPeriodEnd); $touched = true;
    }

    if ($touched) {
      $user->save();
      $this->logger->info('Persisted subscription fields for user @uid', ['@uid' => $user->id()]);
    }
  }

  /**
   * Persist the checkout mode ('payment' | 'subscription' | 'setup') if a field is configured.
   */
  private function persistCheckoutMode(UserInterface $user, ?string $mode): void {
    $field = $this->fieldCheckoutMode();
    if ($mode && $field && $user->hasField($field)) {
      $user->set($field, $mode);
      $user->save();
      $this->logger->info('Set checkout mode for user @uid to @mode', [
        '@uid' => $user->id(),
        '@mode' => $mode,
      ]);
    }
  }
}
