<?php

declare(strict_types=1);

namespace Drupal\stripe_sync\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Stripe\StripeClient;

final class ResyncUserForm extends ConfirmFormBase {

  public function __construct(
    private readonly EntityTypeManagerInterface $etm,
    private readonly ConfigFactoryInterface $cfg,   // keep alias to avoid parent property name
    private readonly LoggerInterface $logger,
  ) {}

  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('logger.channel.stripe_sync'),
    );
  }

  /** @var \Drupal\user\UserInterface|null */
  private ?UserInterface $user = null;

  public function getFormId(): string {
    return 'stripe_sync_resync_user';
  }

  public function getQuestion() {
    $name = $this->user ? $this->user->getDisplayName() : '';
    return $this->t('Re-sync @name from Stripe?', ['@name' => $name]);
  }

  public function getCancelUrl(): Url {
    return Url::fromRoute('entity.user.canonical', ['user' => $this->user?->id()]);
  }

  public function getConfirmText() {
    return $this->t('Re-sync now');
  }

  /**
   * Build with the routed user parameter.
   */
  public function buildForm(array $form, FormStateInterface $form_state, UserInterface $user = null): array {
    $this->user = $user;

    $cfgMod = $this->cfg->get('stripe_sync.settings');
    $fieldCustomer = (string) ($cfgMod->get('field_customer_id') ?? 'field_stripe_customer_id');

    $cid = ($user && $user->hasField($fieldCustomer)) ? (string) ($user->get($fieldCustomer)->value ?? '') : '';

    $build = parent::buildForm($form, $form_state);
    $build['intro'] = [
      '#markup' => '<p>' . $this->t('This will fetch the latest subscription or one-time payment from Stripe and update roles/fields on this Drupal account.') . '</p>',
    ];
    $build['current'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Current mapping'),
      '#items' => [
        $this->t('Email: @mail', ['@mail' => $user->getEmail() ?: '']),
        $this->t('Stripe Customer ID: @cid', ['@cid' => $cid ?: $this->t('(none)')]),
      ],
    ];

    if ($cid === '') {
      $this->messenger()->addWarning($this->t('This user is not linked to a Stripe customer (no cus_… stored). Re-sync will mark them Inactive.'));
    }

    return $build;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $user = $this->user;
    if (!$user instanceof UserInterface) {
      $this->messenger()->addError($this->t('User not found.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    $cfgMod = $this->cfg->get('stripe_sync.settings');
    $roleActive   = (string) ($cfgMod->get('role_active')   ?? 'stripe_member');
    $rolePastDue  = (string) ($cfgMod->get('role_past_due') ?? 'stripe_member_past_due');
    $roleInactive = (string) ($cfgMod->get('role_inactive') ?? 'stripe_member_inactive');

    $fieldCustomer = (string) ($cfgMod->get('field_customer_id')          ?? 'field_stripe_customer_id');
    $fieldSubId    = (string) ($cfgMod->get('field_subscription_id')      ?? 'field_stripe_subscription_id');
    $fieldStatus   = (string) ($cfgMod->get('field_subscription_status')  ?? 'field_subscription_status');
    $fieldExpires  = (string) ($cfgMod->get('field_subscription_expires') ?? 'field_subscription_expires');
    $fieldMode     = (string) ($cfgMod->get('field_checkout_mode')        ?? 'field_checkout_mode');

    $cid = $user->hasField($fieldCustomer) ? (string) ($user->get($fieldCustomer)->value ?? '') : '';

    // Load Stripe secret.
    $cfgStripe = $this->cfg->get('stripe.settings');
    $env = (string) ($cfgStripe->get('environment') ?? 'test'); // 'test'|'live'
    $keyPath = $env === 'live' ? 'apikey.live.secret' : 'apikey.test.secret';
    $secret = (string) ($cfgStripe->get($keyPath) ?? '');
    if ($secret === '' && getenv('STRIPE_SECRET_KEY')) {
      $secret = (string) getenv('STRIPE_SECRET_KEY');
    }
    if ($secret === '') {
      $this->messenger()->addError($this->t('Stripe secret key is not configured.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    try {
      $stripe = new StripeClient($secret);

      if ($cid === '') {
        // No customer: mark inactive & clear fields.
        $this->setRoles($user, [$roleInactive], [$roleActive, $rolePastDue]);
        if ($fieldStatus && $user->hasField($fieldStatus)) {
          $user->set($fieldStatus, 'none');
        }
        if ($fieldSubId && $user->hasField($fieldSubId)) {
          $user->set($fieldSubId, '');
        }
        if ($fieldExpires && $user->hasField($fieldExpires)) {
          $user->set($fieldExpires, NULL);
        }
        if ($fieldMode && $user->hasField($fieldMode)) {
          $user->set($fieldMode, 'none');
        }
        $user->save();
        $this->messenger()->addStatus($this->t('No Stripe customer found. User set to Inactive.'));
        $form_state->setRedirectUrl($this->getCancelUrl());
        return;
      }

      // 1) Try latest subscription (any status).
      $subs = $stripe->subscriptions->all([
        'customer' => $cid,
        'limit' => 1,
        'status' => 'all',
      ]);

      if (!empty($subs->data)) {
        $sub = $subs->data[0];
        $status = (string) $sub->status;
        $subId = (string) $sub->id;
        $periodEnd = isset($sub->current_period_end) ? (int) $sub->current_period_end : NULL;

        if ($fieldSubId && $user->hasField($fieldSubId)) {
          $user->set($fieldSubId, $subId);
        }
        if ($fieldStatus && $user->hasField($fieldStatus)) {
          $user->set($fieldStatus, $status);
        }
        if ($periodEnd && $fieldExpires && $user->hasField($fieldExpires)) {
          $user->set($fieldExpires, $periodEnd);
        }
        if ($fieldMode && $user->hasField($fieldMode)) {
          $user->set($fieldMode, 'subscription');
        }

        // Roles based on status.
        if (in_array($status, ['active', 'trialing'], true)) {
          $this->setRoles($user, [$roleActive], [$rolePastDue, $roleInactive]);
        } elseif (in_array($status, ['past_due', 'unpaid'], true)) {
          $this->setRoles($user, [$rolePastDue], [$roleInactive]);
        } else {
          // includes 'incomplete', 'canceled', etc.
          $this->setRoles($user, [$roleInactive], [$roleActive, $rolePastDue]);
        }

        $user->save();
        $this->messenger()->addStatus($this->t('Re-synced subscription: status=@s, subscription=@sub', [
          '@s' => $status,
          '@sub' => $subId,
        ]));
        $form_state->setRedirectUrl($this->getCancelUrl());
        return;
      }

      // 2) No subscription. Look for a PAID invoice (one-time) and its payment_intent.
      $invoices = $stripe->invoices->all([
        'customer' => $cid,
        'limit' => 1,
      ]);

      if (!empty($invoices->data)) {
        $inv = $invoices->data[0];
        $invStatus = (string) ($inv->status ?? '');
        $invSub = $inv->subscription ?? null;
        $invPi = $inv->payment_intent ?? null;

        // If invoice is paid and NOT tied to a subscription => treat as one-time.
        if ($invStatus === 'paid' && (empty($invSub) || !is_string($invSub))) {
          $accessDays = 0;
          $piSucceeded = false;

          if (is_string($invPi) && $invPi !== '') {
            $pi = $stripe->paymentIntents->retrieve($invPi);
            $piSucceeded = (($pi->status ?? '') === 'succeeded');
            if (!empty($pi->metadata?->access_days)) {
              $accessDays = (int) $pi->metadata->access_days;
            }

            // Apply expires from PI metadata if available.
            if ($accessDays > 0 && $fieldExpires && $user->hasField($fieldExpires)) {
              $created = (int) ($pi->created ?? time());
              $user->set($fieldExpires, $created + ($accessDays * 86400));
            }
          }

          if ($piSucceeded) {
            // Grant Active and mark mode.
            $this->setRoles($user, [$roleActive], [$roleInactive]);
            if ($fieldMode && $user->hasField($fieldMode)) {
              $user->set($fieldMode, 'payment');
            }
            // Clear sub fields because this was a one-time.
            if ($fieldSubId && $user->hasField($fieldSubId)) {
              $user->set($fieldSubId, '');
            }
            if ($fieldStatus && $user->hasField($fieldStatus)) {
              $user->set($fieldStatus, 'none');
            }

            $user->save();
            $this->messenger()->addStatus($this->t('Re-synced one-time payment: invoice paid, access ensured.'));
            $form_state->setRedirectUrl($this->getCancelUrl());
            return;
          }
        }
      }

      // 3) Fallback: check latest PaymentIntent directly (customer-scoped).
      $pis = $stripe->paymentIntents->all([
        'customer' => $cid,
        'limit' => 1,
      ]);

      if (!empty($pis->data)) {
        $pi = $pis->data[0];
        if (($pi->status ?? '') === 'succeeded') {
          $accessDays = 0;
          if (!empty($pi->metadata?->access_days)) {
            $accessDays = (int) $pi->metadata->access_days;
          }
          if ($accessDays > 0 && $fieldExpires && $user->hasField($fieldExpires)) {
            $created = (int) ($pi->created ?? time());
            $user->set($fieldExpires, $created + ($accessDays * 86400));
          }

          // Grant Active and mark mode.
          $this->setRoles($user, [$roleActive], [$roleInactive]);
          if ($fieldMode && $user->hasField($fieldMode)) {
            $user->set($fieldMode, 'payment');
          }
          // Clear sub fields for one-time.
          if ($fieldSubId && $user->hasField($fieldSubId)) {
            $user->set($fieldSubId, '');
          }
          if ($fieldStatus && $user->hasField($fieldStatus)) {
            $user->set($fieldStatus, 'none');
          }

          $user->save();
          $this->messenger()->addStatus($this->t('Re-synced one-time payment: latest payment succeeded.'));
          $form_state->setRedirectUrl($this->getCancelUrl());
          return;
        }
      }

      // 4) Nothing paid found: mark inactive & clear.
      $this->setRoles($user, [$roleInactive], [$roleActive, $rolePastDue]);
      if ($fieldStatus && $user->hasField($fieldStatus)) {
        $user->set($fieldStatus, 'none');
      }
      if ($fieldSubId && $user->hasField($fieldSubId)) {
        $user->set($fieldSubId, '');
      }
      if ($fieldMode && $user->hasField($fieldMode)) {
        $user->set($fieldMode, 'none');
      }
      // Do not touch expires here; leave as-is unless you want to clear:
      // if ($fieldExpires && $user->hasField($fieldExpires)) { $user->set($fieldExpires, NULL); }

      $user->save();
      $this->messenger()->addStatus($this->t('No active subscription or successful one-time payment found. User set to Inactive.'));
    }
    catch (\Throwable $e) {
      $this->logger->error('Stripe re-sync failed for user @uid: @msg', ['@uid' => $user->id(), '@msg' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Stripe re-sync failed: @msg', ['@msg' => $e->getMessage()]));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  private function setRoles(UserInterface $user, array $add, array $remove): void {
    $changed = false;
    foreach ($add as $rid) {
      if ($rid && !$user->hasRole($rid)) { $user->addRole($rid); $changed = true; }
    }
    foreach ($remove as $rid) {
      if ($rid && $user->hasRole($rid)) { $user->removeRole($rid); $changed = true; }
    }
    if ($changed) {
      $user->save();
    }
  }
}
