<?php

declare(strict_types=1);

namespace Drupal\stripe_sync\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;
use Drupal\field\Entity\FieldConfig;

/**
 * Settings form for Stripe sync.
 */
final class SettingsForm extends ConfigFormBase {

  public function getFormId(): string {
    return 'stripe_sync_settings';
  }

  protected function getEditableConfigNames(): array {
    return ['stripe_sync.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $cfg = $this->config('stripe_sync.settings');

    // ───────────────────────────────────────────────────────────────────────────
    // Collect roles.
    // ───────────────────────────────────────────────────────────────────────────
    $roles = [];
    /** @var \Drupal\user\Entity\Role[] $role_entities */
    $role_entities = Role::loadMultiple();
    foreach ($role_entities as $rid => $role) {
      $roles[$rid] = $role->label();
    }
    ksort($roles);

    // ───────────────────────────────────────────────────────────────────────────
    // Collect user fields on user bundle.
    // ───────────────────────────────────────────────────────────────────────────
    $field_opts = ['' => '- None -'];
    $storage = \Drupal::entityTypeManager()->getStorage('field_config');
    /** @var \Drupal\field\Entity\FieldConfig[] $user_fields */
    $user_fields = $storage->loadByProperties([
      'entity_type' => 'user',
      'bundle' => 'user',
    ]);
    foreach ($user_fields as $f) {
      if ($f instanceof FieldConfig && !$f->isDeleted()) {
        $field_opts[$f->getName()] = $f->label() . ' (' . $f->getName() . ')';
      }
    }
    ksort($field_opts);

    // ───────────────────────────────────────────────────────────────────────────
    // Roles
    // ───────────────────────────────────────────────────────────────────────────
    $form['roles'] = [
      '#type' => 'details',
      '#title' => $this->t('Roles'),
      '#open' => TRUE,
    ];
    $form['roles']['role_active'] = [
      '#type' => 'select',
      '#title' => $this->t('Active role'),
      '#options' => $roles,
      '#default_value' => $cfg->get('role_active') ?? 'stripe_member',
      '#required' => TRUE,
    ];
    $form['roles']['role_past_due'] = [
      '#type' => 'select',
      '#title' => $this->t('Past due role'),
      '#options' => ['' => '- None -'] + $roles,
      '#default_value' => $cfg->get('role_past_due') ?? 'stripe_member_past_due',
    ];
    $form['roles']['role_inactive'] = [
      '#type' => 'select',
      '#title' => $this->t('Inactive role'),
      '#options' => $roles,
      '#default_value' => $cfg->get('role_inactive') ?? 'stripe_member_inactive',
      '#required' => TRUE,
    ];

    // ───────────────────────────────────────────────────────────────────────────
    // User fields
    // ───────────────────────────────────────────────────────────────────────────
    $form['fields'] = [
      '#type' => 'details',
      '#title' => $this->t('User fields'),
      '#open' => TRUE,
    ];
    $form['fields']['field_customer_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Stripe Customer ID field'),
      '#options' => $field_opts,
      '#default_value' => $cfg->get('field_customer_id') ?? 'field_stripe_customer_id',
      '#required' => TRUE,
    ];
    $form['fields']['field_subscription_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Subscription ID field (optional)'),
      '#options' => $field_opts,
      '#default_value' => $cfg->get('field_subscription_id') ?? 'field_stripe_subscription_id',
    ];
    $form['fields']['field_subscription_status'] = [
      '#type' => 'select',
      '#title' => $this->t('Subscription status field (optional)'),
      '#options' => $field_opts,
      '#default_value' => $cfg->get('field_subscription_status') ?? 'field_subscription_status',
    ];
    $form['fields']['field_subscription_expires'] = [
      '#type' => 'select',
      '#title' => $this->t('Expires (timestamp) field (optional)'),
      '#options' => $field_opts,
      '#default_value' => $cfg->get('field_subscription_expires') ?? 'field_subscription_expires',
      '#description' => $this->t('For subscriptions: stores current_period_end. For one-time: stores now + access_days.'),
    ];
    $form['fields']['field_checkout_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Checkout mode field (optional)'),
      '#options' => $field_opts,
      '#default_value' => $cfg->get('field_checkout_mode') ?? 'field_checkout_mode',
      '#description' => $this->t('Stores last checkout mode: payment | subscription | setup.'),
    ];

    // ───────────────────────────────────────────────────────────────────────────
    // Checkout options
    // ───────────────────────────────────────────────────────────────────────────
    $form['checkout'] = [
      '#type' => 'details',
      '#title' => $this->t('Checkout options'),
      '#open' => TRUE,
    ];
    $form['checkout']['allow_payment_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow one-time purchases (Checkout mode=payment)'),
      '#default_value' => (bool) ($cfg->get('allow_payment_mode') ?? TRUE),
      '#description' => $this->t('If disabled, only subscription prices may be used.'),
    ];
    $form['checkout']['require_access_days_for_payment'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require "access_days" metadata for one-time prices'),
      '#default_value' => (bool) ($cfg->get('require_access_days_for_payment') ?? FALSE),
      '#description' => $this->t('When enabled, one-time prices must define access_days on the Stripe Price (or Product fallback).'),
      '#states' => [
        'visible' => [
          ':input[name="allow_payment_mode"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['checkout']['enable_invoice_creation_for_payment'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create an invoice for one-time purchases'),
      '#default_value' => (bool) ($cfg->get('enable_invoice_creation_for_payment') ?? FALSE),
      '#description' => $this->t('If enabled, Stripe will create a paid invoice after one-time Checkout.'),
      '#states' => [
        'visible' => [
          ':input[name="allow_payment_mode"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // ───────────────────────────────────────────────────────────────────────────
    // Duplicate-subscription policy
    // ───────────────────────────────────────────────────────────────────────────
    $form['dup'] = [
      '#type' => 'details',
      '#title' => $this->t('Prevent duplicate subscriptions'),
      '#open' => TRUE,
    ];
    $form['dup']['prevent_duplicate_subscriptions'] = [
      '#type' => 'radios',
      '#title' => $this->t('Policy'),
      '#options' => [
        'none' => $this->t('Do not block'),
        'same_product' => $this->t('Block if user already has a non-terminal subscription for the same product'),
        'any' => $this->t('Block if user has any non-terminal subscription'),
      ],
      '#default_value' => $cfg->get('prevent_duplicate_subscriptions') ?? 'same_product',
      '#description' => $this->t('Non-terminal includes active, trialing, past_due, unpaid, incomplete, paused. Terminal: canceled, incomplete_expired.'),
    ];

    // ───────────────────────────────────────────────────────────────────────────
    // Daily sync (cron)
    // ───────────────────────────────────────────────────────────────────────────
    $form['cron'] = [
      '#type' => 'details',
      '#title' => $this->t('Daily sync (cron)'),
      '#open' => TRUE,
    ];
    $form['cron']['enable_daily_sync'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable daily Stripe → Drupal sync'),
      '#default_value' => (bool) ($cfg->get('enable_daily_sync') ?? FALSE),
      '#description' => $this->t('Keeps roles/fields aligned if webhooks are missed. Runs once a day at the time below.'),
    ];

    // Time pickers (24h).
    $hour_options = [];
    for ($h = 0; $h <= 23; $h++) {
      $hour_options[$h] = sprintf('%02d', $h);
    }
    $min_options = [];
    for ($m = 0; $m <= 55; $m += 5) {
      $min_options[$m] = sprintf('%02d', $m);
    }

    $form['cron']['daily_sync_hour'] = [
      '#type' => 'select',
      '#title' => $this->t('Hour (24h)'),
      '#options' => $hour_options,
      '#default_value' => (int) ($cfg->get('daily_sync_hour') ?? 3),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cron']['daily_sync_minute'] = [
      '#type' => 'select',
      '#title' => $this->t('Minute'),
      '#options' => $min_options,
      '#default_value' => (int) ($cfg->get('daily_sync_minute') ?? 0),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cron']['daily_sync_batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch size per run'),
      '#default_value' => (int) ($cfg->get('daily_sync_batch_size') ?? 100),
      '#min' => 1,
      '#max' => 1000,
      '#description' => $this->t('How many users to process in one cron run (queue will continue on next runs).'),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cron']['daily_sync_max_requests'] = [
      '#type' => 'number',
      '#title' => $this->t('Max Stripe API requests per run'),
      '#default_value' => (int) ($cfg->get('daily_sync_max_requests') ?? 200),
      '#min' => 10,
      '#max' => 2000,
      '#description' => $this->t('Safety cap to avoid hitting rate limits.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cron']['include_payment_mode_users'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Also sweep one-time (mode=payment) access windows'),
      '#default_value' => (bool) ($cfg->get('include_payment_mode_users') ?? TRUE),
      '#description' => $this->t('If checked, cron will expire access when "Expires (timestamp)" is in the past.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cron']['include_subid_without_customer'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include users with Subscription ID but no Customer ID'),
      '#default_value' => (bool) ($cfg->get('include_subid_without_customer') ?? FALSE),
      '#description' => $this->t('Edge case support; may require extra Stripe lookups.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_daily_sync"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Read-only last run information.
    $last_run = (int) ($cfg->get('daily_sync_last_run') ?? 0);
    $form['cron']['daily_sync_last_run'] = [
      '#type' => 'item',
      '#title' => $this->t('Last run'),
      '#markup' => $last_run ? date('D, d M Y - H:i', $last_run) : $this->t('Never'),
    ];

    // ───────────────────────────────────────────────────────────────────────────
    // Actions: Save + Run now
    // ───────────────────────────────────────────────────────────────────────────
    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];
    $form['actions']['run_now'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run daily sync now'),
      '#submit' => ['::runNowSubmit'],
      // Allow running without validating/saving the form.
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['button', 'button--secondary']],
    ];

    return parent::buildForm($form, $form_state);
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    if ($form_state->getValue('enable_daily_sync')) {
      $bs = (int) $form_state->getValue('daily_sync_batch_size');
      if ($bs < 1 || $bs > 1000) {
        $form_state->setErrorByName('daily_sync_batch_size', $this->t('Batch size must be between 1 and 1000.'));
      }
      $mr = (int) $form_state->getValue('daily_sync_max_requests');
      if ($mr < 10 || $mr > 2000) {
        $form_state->setErrorByName('daily_sync_max_requests', $this->t('Max Stripe requests must be between 10 and 2000.'));
      }
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $this->config('stripe_sync.settings')
      // Roles.
      ->set('role_active', $form_state->getValue('role_active'))
      ->set('role_past_due', $form_state->getValue('role_past_due'))
      ->set('role_inactive', $form_state->getValue('role_inactive'))
      // Fields.
      ->set('field_customer_id', $form_state->getValue('field_customer_id'))
      ->set('field_subscription_id', $form_state->getValue('field_subscription_id'))
      ->set('field_subscription_status', $form_state->getValue('field_subscription_status'))
      ->set('field_subscription_expires', $form_state->getValue('field_subscription_expires'))
      ->set('field_checkout_mode', $form_state->getValue('field_checkout_mode'))
      // Checkout options.
      ->set('allow_payment_mode', (bool) $form_state->getValue('allow_payment_mode'))
      ->set('require_access_days_for_payment', (bool) $form_state->getValue('require_access_days_for_payment'))
      ->set('enable_invoice_creation_for_payment', (bool) $form_state->getValue('enable_invoice_creation_for_payment'))
      // Duplicate-subscription policy.
      ->set('prevent_duplicate_subscriptions', (string) $form_state->getValue('prevent_duplicate_subscriptions'))
      // Daily sync (cron).
      ->set('enable_daily_sync', (bool) $form_state->getValue('enable_daily_sync'))
      ->set('daily_sync_hour', (int) $form_state->getValue('daily_sync_hour'))
      ->set('daily_sync_minute', (int) $form_state->getValue('daily_sync_minute'))
      ->set('daily_sync_batch_size', (int) $form_state->getValue('daily_sync_batch_size'))
      ->set('daily_sync_max_requests', (int) $form_state->getValue('daily_sync_max_requests'))
      ->set('include_payment_mode_users', (bool) $form_state->getValue('include_payment_mode_users'))
      ->set('include_subid_without_customer', (bool) $form_state->getValue('include_subid_without_customer'))
      ->save();
  }

  /**
   * Submit handler: enqueue a daily batch immediately (no validation/no save).
   */
  public function runNowSubmit(array &$form, FormStateInterface $form_state): void {
    // Ensure our helper from the .module file exists.
    if (!function_exists('stripe_sync_enqueue_daily_batch')) {
      $this->messenger()->addError($this->t('Run helper not found. Clear caches and verify the module file is loaded.'));
      return;
    }

    $count = stripe_sync_enqueue_daily_batch(TRUE);
    // Mark "last run" now so scheduled cron won’t double-run today.
    if (function_exists('_stripe_sync_mark_daily_run')) {
      _stripe_sync_mark_daily_run(\Drupal::time()->getRequestTime());
    }

    $this->messenger()->addStatus($this->t('Queued @n users for daily sync.', ['@n' => max(0, (int) $count)]));
  }
}
