<?php

declare(strict_types=1);

namespace Drupal\stripe_sync\Plugin\QueueWorker;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Stripe\StripeClient;

/**
 * Processes daily Stripe re-sync for users.
 *
 * @QueueWorker(
 *   id = "stripe_sync.daily",
 *   title = @Translation("Stripe Daily Sync"),
 *   cron = {
 *     "time" = 60
 *   }
 * )
 */
final class StripeDailySyncWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $etm,
    private readonly ConfigFactoryInterface $cfg,
    private readonly LoggerChannelInterface $logger,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('logger.channel.stripe_sync'),
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    // Expect an array like ['uid' => 123].
    $uid = (int) ($data['uid'] ?? 0);
    if ($uid <= 0) {
      $this->logger->warning('Queue item missing uid.');
      return;
    }

    /** @var \Drupal\user\UserInterface|null $user */
    $user = $this->etm->getStorage('user')->load($uid);
    if (!$user instanceof UserInterface) {
      $this->logger->warning('Queue item refers to missing user @uid.', ['@uid' => $uid]);
      return;
    }

    $cfgMod = $this->cfg->get('stripe_sync.settings');

    // Roles/fields from settings.
    $roleActive   = (string) ($cfgMod->get('role_active')   ?? 'stripe_member');
    $rolePastDue  = (string) ($cfgMod->get('role_past_due') ?? 'stripe_member_past_due');
    $roleInactive = (string) ($cfgMod->get('role_inactive') ?? 'stripe_member_inactive');

    $fCustomer = (string) ($cfgMod->get('field_customer_id')          ?? 'field_stripe_customer_id');
    $fSubId    = (string) ($cfgMod->get('field_subscription_id')      ?? 'field_stripe_subscription_id');
    $fStatus   = (string) ($cfgMod->get('field_subscription_status')  ?? 'field_subscription_status');
    $fExpires  = (string) ($cfgMod->get('field_subscription_expires') ?? 'field_subscription_expires');
    $fMode     = (string) ($cfgMod->get('field_checkout_mode')        ?? '');

    $includePaymentSweep = (bool) ($cfgMod->get('include_payment_mode_users') ?? TRUE);

    // Stripe secret (from the stripe module config).
    $cfgStripe = $this->cfg->get('stripe.settings');
    $env = (string) ($cfgStripe->get('environment') ?? 'test');
    $keyPath = $env === 'live' ? 'apikey.live.secret' : 'apikey.test.secret';
    $secret = (string) ($cfgStripe->get($keyPath) ?? '');
    if ($secret === '' && getenv('STRIPE_SECRET_KEY')) {
      $secret = (string) getenv('STRIPE_SECRET_KEY');
    }
    if ($secret === '') {
      $this->logger->error('Daily sync: Stripe secret key not configured.');
      return;
    }

    $cid = ($fCustomer && $user->hasField($fCustomer)) ? (string) ($user->get($fCustomer)->value ?? '') : '';

    try {
      $stripe = new StripeClient($secret);

      // If we have a Stripe customer ID, prefer live subscription data.
      if ($cid !== '') {
        $subs = $stripe->subscriptions->all([
          'customer' => $cid,
          'limit' => 1,
          'status' => 'all',
        ]);

        if (!empty($subs->data)) {
          $sub = $subs->data[0];
          $status = (string) $sub->status;
          $subId = (string) $sub->id;
          $periodEnd = isset($sub->current_period_end) ? (int) $sub->current_period_end : NULL;

          if ($fSubId && $user->hasField($fSubId)) {
            $user->set($fSubId, $subId);
          }
          if ($fStatus && $user->hasField($fStatus)) {
            $user->set($fStatus, $status);
          }
          if ($periodEnd && $fExpires && $user->hasField($fExpires)) {
            $user->set($fExpires, $periodEnd);
          }
          // NEW: mark checkout mode = 'subscription'.
          if ($fMode && $user->hasField($fMode)) {
            $user->set($fMode, 'subscription');
          }

          // Roles from status.
          if (in_array($status, ['active', 'trialing'], true)) {
            $this->setRoles($user, [$roleActive], [$rolePastDue, $roleInactive]);
          }
          elseif (in_array($status, ['past_due', 'unpaid'], true)) {
            $this->setRoles($user, $rolePastDue ? [$rolePastDue] : [], [$roleInactive]);
          }
          else {
            // incomplete, canceled, paused, etc.
            $this->setRoles($user, [$roleInactive], [$roleActive, $rolePastDue]);
          }

          $user->save();
          $this->logger->info('Daily sync updated user @uid from subscription @sub (@status).', [
            '@uid' => $user->id(),
            '@sub' => $subId,
            '@status' => $status,
          ]);
          return;
        }
        else {
          // No active/any subscriptions for this customer -> fall through.
          $this->logger->notice('Daily sync: user @uid has customer @cid but no subscriptions.', [
            '@uid' => $user->id(),
            '@cid' => $cid,
          ]);
        }
      }

      // No customer or no subs: optionally handle one-time access expiry.
      if ($includePaymentSweep && $fExpires && $user->hasField($fExpires)) {
        $expires = (int) ($user->get($fExpires)->value ?? 0);

        if ($expires > 0) {
          // NEW: mark checkout mode = 'payment'.
          if ($fMode && $user->hasField($fMode)) {
            $user->set($fMode, 'payment');
          }

          if ($expires < \Drupal::time()->getRequestTime()) {
            // Expired -> mark inactive.
            if ($fStatus && $user->hasField($fStatus)) {
              $user->set($fStatus, 'expired');
            }
            $this->setRoles($user, [$roleInactive], [$roleActive, $rolePastDue]);
            $user->save();
            $this->logger->info('Daily sync: expired one-time access for user @uid.', ['@uid' => $user->id()]);
            return;
          }
          else {
            // Still valid -> ensure active.
            if ($fStatus && $user->hasField($fStatus)) {
              $user->set($fStatus, 'active_one_time');
            }
            $this->setRoles($user, [$roleActive], [$roleInactive]);
            $user->save();
            $this->logger->info('Daily sync: one-time access still valid for user @uid.', ['@uid' => $user->id()]);
            return;
          }
        }
      }

      // Default: nothing to change.
      $this->logger->debug('Daily sync: no changes for user @uid.', ['@uid' => $user->id()]);
    }
    catch (\Throwable $e) {
      // Do not requeue forever; just log.
      $this->logger->error('Daily sync failed for user @uid: @msg', [
        '@uid' => $user->id(),
        '@msg' => $e->getMessage(),
      ]);
    }
  }

  private function setRoles(UserInterface $user, array $add, array $remove): void {
    $changed = FALSE;
    foreach ($add as $rid) {
      if ($rid && !$user->hasRole($rid)) { $user->addRole($rid); $changed = TRUE; }
    }
    foreach ($remove as $rid) {
      if ($rid && $user->hasRole($rid)) { $user->removeRole($rid); $changed = TRUE; }
    }
    if ($changed) {
      $user->save();
    }
  }

}
