<?php

declare(strict_types=1);

namespace Drupal\style_options_spacing\Plugin\StyleOption;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormStateInterface;

/**
 * Define the margin style option plugin.
 *
 * @StyleOption(
 *   id = "spacing_margin",
 *   label = @Translation("Style: Margin Space")
 * )
 */
class StyleOptionSpacingMargin extends StyleOptionSpacingPresetBase {

  /**
   * AJAX handler.
   */
  public function onTypeChanged(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand(
      '.style-option-margin',
      $form['behavior_plugins']['style_options']['spacing_margin']['margin']['value'] ?? [],
    ));
    return $response;
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // For AJAX requests replace the default values.
    $values = $this->getValue('margin') ?? ['type' => NULL];

    // @TODO: Change this once $form_state get values can be used.
    $submitted_values = $form_state->getUserInput()['behavior_plugins']['style_options']['spacing_margin']['margin'] ?? [];
    if (!empty($submitted_values)) {
      $values['type'] = $submitted_values['type'] ?? NULL;
    }

    // Builds the margin fields.
    $form['margin'] = [
      '#tree' => TRUE,
      '#type' => 'fieldset',
      '#title' => $this->t('Margin'),
      '#description' => $this->t('Set the margin for this element.'),
      '#description_display' => 'before',
      'type' => [
        '#ajax' => [
          'callback' => [$this, 'onTypeChanged'],
          'event' => 'change',
        ],
        '#default_value' => $values['type'],
        '#description' => $this->t('Select the type of margin to apply.'),
        '#description_display' => 'before',
        '#empty_option' => $this->t('Set the margin options'),
        '#options' => [
          'presets' => $this->t('Presets'),
          'free' => $this->t('Free value'),
        ],
        '#title' => $this->t('Type'),
        '#type' => 'select',
      ]
    ];

    // Builds the fields depending on selected option.
    switch ($values['type']) {
      case 'presets':
        $form['margin']['value'] = $this->getPresetSpacingOptionsFieldForm($values['value'] ?? [], 'margin');
        break;

      case 'free':
        $form['margin']['value'] = $this->getFreeSpacingFieldForm($values['value'] ?? []);
        break;
    }

    // Sets option if available.
    if (isset($form['margin']['type'])) {
      $form['margin']['value']['#prefix'] = '<div class="style-option-margin">';
      $form['margin']['value']['#suffix'] = '</div>';
    }

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function build(array $build, $value = '') {
    $margin = $this->getValue('margin') ?? [];
    if (empty($margin) || empty($margin['value'])) {
      return $build;
    }

    foreach ($margin['value'] as $group => $directions) {
      foreach ($directions as $dir => $val) {
        if (empty($val)) {
          continue;
        }

        if (($margin['type'] ?? NULL) == 'free') {
          $build['#attributes']['style'][] = "margin-{$dir}: {$val};";
        }
        else {
          $build['#attributes']['class'][] = $val;
        }
      }
    }

    return $build;
  }

}
