<?php

declare(strict_types=1);

namespace Drupal\style_options_spacing\Plugin\StyleOption;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormStateInterface;

/**
 * Define the padding style option plugin.
 *
 * @StyleOption(
 *   id = "spacing_padding",
 *   label = @Translation("Style: Padding Space")
 * )
 */
class StyleOptionSpacingPadding extends StyleOptionSpacingPresetBase {

  /**
   * AJAX handler.
   */
  public function onTypeChanged(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand(
      '.style-option-padding',
      $form['behavior_plugins']['style_options']['spacing_padding']['padding']['value'] ?? [],
    ));
    return $response;
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // For AJAX requests replace the default values.
    $values = $this->getValue('padding') ?? ['type' => NULL];

    // @TODO: Change this once $form_state get values can be used.
    $submitted_values = $form_state->getUserInput()['behavior_plugins']['style_options']['spacing_padding']['padding'] ?? [];
    if (!empty($submitted_values)) {
      $values['type'] = $submitted_values['type'] ?? NULL;
    }

    // Builds the padding fields.
    $form['padding'] = [
      '#tree' => TRUE,
      '#type' => 'fieldset',
      '#title' => $this->t('Padding'),
      '#description' => $this->t('Set the padding for this element.'),
      '#description_display' => 'before',
      'type' => [
        '#ajax' => [
          'callback' => [$this, 'onTypeChanged'],
          'event' => 'change',
        ],
        '#default_value' => $values['type'],
        '#description' => $this->t('Select the type of padding to apply.'),
        '#empty_option' => $this->t('Set the padding options'),
        '#options' => [
          'presets' => $this->t('Presets'),
          'free' => $this->t('Free value'),
        ],
        '#title' => $this->t('Type'),
        '#type' => 'select',
      ]
    ];

    // Builds the fields depending on selected option.
    switch ($values['type']) {
      case 'presets':
        $form['padding']['value'] = $this->getPresetSpacingOptionsFieldForm($values['value'] ?? [], 'padding');
        break;

      case 'free':
        $form['padding']['value'] = $this->getFreeSpacingFieldForm($values['value'] ?? []);
        break;
    }

    // Sets option if available.
    if (isset($form['padding']['type'])) {
      $form['padding']['value']['#prefix'] = '<div class="style-option-padding">';
      $form['padding']['value']['#suffix'] = '</div>';
    }

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function build(array $build, $value = '') {
    $padding = $this->getValue('padding') ?? [];
    if (empty($padding) || empty($padding['value'])) {
      return $build;
    }

    foreach ($padding['value'] as $group => $directions) {
      foreach ($directions as $dir => $val) {
        if (empty($val)) {
          continue;
        }

        if (($padding['type'] ?? NULL) == 'free') {
          $build['#attributes']['style'][] = "padding-{$dir}: {$val};";
        }
        else {
          $build['#attributes']['class'][] = $val;
        }
      }
    }

    return $build;
  }

}
