<?php

declare(strict_types=1);

namespace Drupal\style_options_spacing\Plugin\StyleOption;

use Drupal\style_options\StyleOptionStyleTrait;
use Drupal\style_options\Plugin\StyleOptionPluginBase;

abstract class StyleOptionSpacingPresetBase extends StyleOptionPluginBase {

  use StyleOptionStyleTrait;

  /**
   * Gets the available presents for the space option.
   *
   * @param string $category
   *  The category of presens to get.
   *
   * @return mixed[]
   *  An array where the key is the preset class and the value its label.s
   */
  private function getPresetSpacingOptions(string $category) : array {
    // Just a basic sample, ideally this is overriden as an option.
    $values = [
      'small' => 1,
      'medium' => 2,
      'large' => 3,
    ];
    $defaults = [];
    foreach (['top', 'right', 'bottom', 'left'] as $dir) {
      foreach ($values as $preset => $label) {
        if (!isset($defaults[$dir])) {
          $defaults[$dir] = [];
        }
        $defaults[$dir]["{$category}-{$dir}-{$preset}"] = $label;
      }
    }

    // Loads the presets from config and replaces the defaults.
    $presetsFromConfig = $this->getConfiguration('presets');
    if (!empty($presetsFromConfig)) {
      foreach ($presetsFromConfig as $dir => $presets) {
        if (isset($defaults[$dir])) {
          $defaults[$dir] = $presets;
        }
      }
    }

    return $defaults;
  }

  /**
   * Gets the preset form fields for the set of options.
   *
   * @param array $defaultValues
   *  The default values that should be checked for each option.
   *
   * @param string $category
   *  The category of presents to get.
   *
   * @return array[]
   */
  protected function getPresetSpacingOptionsFieldForm(array $defaultValues, string $category) : array {
    $spacingOptions = $this->getPresetSpacingOptions($category);
    return [
      'vertical' => [
        '#description' => $this->t('A spacing preset.'),
        '#description_display' => 'before',
        '#type' => 'fieldset',
        '#title' => $this->t('Vertical'),
        'top' => [
          '#empty_option' => $this->t('None'),
          '#default_value' => $defaultValues['vertical']['top'] ?? NULL,
          '#options' => $spacingOptions['top'],
          '#type' => 'select',
          '#title' => $this->t('Top'),
        ],
        'bottom' => [
          '#empty_option' => $this->t('None'),
          '#default_value' => $defaultValues['vertical']['bottom'] ?? NULL,
          '#options' => $spacingOptions['bottom'],
          '#type' => 'select',
          '#title' => $this->t('Bottom'),
        ],
      ],
      'horizontal' => [
        '#description' => $this->t('A spacing preset.'),
        '#description_display' => 'before',
        '#type' => 'fieldset',
        '#title' => $this->t('Horizontal'),
        'left' => [
          '#default_value' => $defaultValues['horizontal']['left'] ?? NULL,
          '#empty_option' => $this->t('None'),
          '#options' => $spacingOptions['left'],
          '#type' => 'select',
          '#title' => $this->t('Left'),
        ],
        'right' => [
          '#default_value' => $defaultValues['horizontal']['right'] ?? NULL,
          '#empty_option' => $this->t('None'),
          '#options' => $spacingOptions['right'],
          '#type' => 'select',
          '#title' => $this->t('Right'),
        ],
      ],
    ];
  }

  /**
   * Gets the spacing form fields for the set of options.
   *
   * @param array $defaultValues
   *  The default values that should be checked for each option.
   *
   * @return array[]
   */
  protected function getFreeSpacingFieldForm(array $defaultValues) : array {
    return [
      'vertical' => [
        '#description' => $this->t('Enter values with the unit, e.g. 10px, 30rem, etc.'),
        '#description_display' => 'before',
        '#type' => 'fieldset',
        '#title' => $this->t('Vertical'),
        'top' => [
          '#default_value' => $defaultValues['vertical']['top'] ?? NULL,
          '#type' => 'textfield',
          '#title' => $this->t('Top'),
        ],
        'bottom' => [
          '#default_value' => $defaultValues['vertical']['bottom'] ?? NULL,
          '#type' => 'textfield',
          '#title' => $this->t('Bottom'),
        ],
      ],
      'horizontal' => [
        '#description' => $this->t('Enter values with the unit, e.g. 10px, 30rem, etc.'),
        '#description_display' => 'before',
        '#type' => 'fieldset',
        '#title' => $this->t('Horizontal'),
        'left' => [
          '#default_value' => $defaultValues['horizontal']['left'] ?? NULL,
          '#type' => 'textfield',
          '#title' => $this->t('Left'),
        ],
        'right' => [
          '#default_value' => $defaultValues['horizontal']['right'] ?? NULL,
          '#type' => 'textfield',
          '#title' => $this->t('Right'),
        ],
      ],
    ];
  }
}
