/**
 * @file
 * Initiates map(s) for the Styled Google Map module.
 *
 * A single or multiple Styled Google Maps will be initiated.
 * Drupal behaviors are used to make sure ajax called map(s) are correctly
 *     loaded.
 */

(function (Drupal) {

  /**
   * Calculates the route and displays it on the map.
   */
  function calculateAndDisplayRoute(directionsService, directionsDisplay, map_id, destination) {
    var selectedMode = document.getElementById('getting_directions_mode-' + map_id).value;
    var address = document.getElementById('getting_directions_address-' + map_id).value;
    if (document.getElementById('getting_directions_location-' + map_id).checked) {
      if (typeof window.drupalSettings.user_current_location === 'object' && typeof window.drupalSettings.user_current_location.getPosition === 'function') {
        address = window.drupalSettings.user_current_location.getPosition();
      }
    }
    directionsService.route({
      origin: address,
      destination: {lat: destination.lat, lng: destination.lon},
      travelMode: google.maps.TravelMode[selectedMode]
    }, function(response, status) {
      if (status === 'OK') {
        directionsDisplay.setDirections(response);
      } else {
        const messages = new Drupal.Message();
        var message = Drupal.t('Directions request failed due to @status', {'@status': status});
        if (typeof Drupal.Message !== 'undefined') {
          messages.add(message, {type: 'error'});
        } else {
          window.alert(message);
        }
      }
    });
  }

  /**
   * Behavior to display the map.
   *
   * @type {{attach: Drupal.behaviors.styled_google_maps.attach}}
   */
  Drupal.behaviors.styled_google_maps = {
    attach: function initMap(context, settings) {
      var maps = settings.styled_google_map;
      var markers = [];
      // There can be multiple maps on a page, let's iterate over them.
      for (var i in maps) {
        var current_map = settings.maps['id' + maps[i]];
        var map_id = current_map.id;
        var map_container = document.getElementById(map_id);
        // Continue in case we find the map object on the current page.
        if (map_container) {
          // Declare the map locations, style, bounds, settings and types.
          var map_locations = current_map.locations;
          var map_settings = current_map.settings;

          // Set the map inline style width and height.
          map_container.style.width = current_map.settings.width;
          map_container.style.height = current_map.settings.height;

          // Disable scrolling on screens smaller than 480 pixels.
          // TODO: This is deprecated and should be removed due to gesture handling.
          map_settings.draggable = document.documentElement.clientWidth > 480 ? map_settings.draggable : map_settings.mobile_draggable;
          const mapId = map_settings?.style?.map_id ? map_settings.style.map_id : map_settings.main.styled_google_map_id;
          // Declare and initialize the base settings of the Google Map.
          var init_map = {
            mapId: mapId,
            gestureHandling: map_settings.gestureHandling,
            zoom: parseInt(map_settings.zoom.default),
            mapTypeId: map_settings.style.maptype.toLowerCase(),
            disableDefaultUI: !map_settings.ui,
            maxZoom: parseInt(map_settings.zoom.max),
            minZoom: parseInt(map_settings.zoom.min),
            mapTypeControl: map_settings.maptypecontrol,
            scaleControl: map_settings.scalecontrol,
            rotateControl: map_settings.rotatecontrol,
            streetViewControl: map_settings.streetviewcontrol,
            zoomControl: map_settings.zoomcontrol,
            fullscreenControl: map_settings.fullscreen,
            draggable: map_settings.draggable
          };
          var map = new google.maps.Map(map_container, init_map);
          var bounds = new google.maps.LatLngBounds();

          var heatmap = {};

          // Additional functionality to enable a heatmap.
          if (typeof map_settings.heat_map != 'undefined' && map_settings.heat_map.heatmap_enabled) {
            var heatmap_data = [];
            for (var k = 0; k < map_settings.heat_map.data.length; k++) {
              var heatmap_item = {
                location: new google.maps.LatLng(map_settings.heat_map.data[k].lat, map_settings.heat_map.data[k].lon),
                weight: map_settings.heat_map.data[k].weight,
              };
              heatmap_data.push(heatmap_item);
            }
            heatmap = new google.maps.visualization.HeatmapLayer({
              data: heatmap_data,
              gradient: map_settings.heat_map.gradient,
              opacity: map_settings.heat_map.opacity,
              maxIntensity: map_settings.heat_map.maxIntensity,
              dissipating: map_settings.heat_map.dissipating,
              radius: map_settings.heat_map.radius
            });
            heatmap.setMap(map);
          }
          else {
            // Loop each location (pin) our map contains.
            for (var j in map_locations) {
              // A marker with a URL pointing to a custom image.
              const icon = document.createElement('img');
              icon.src = map_locations[j].pin;
              if (map_settings.style.pin_width !== '') {
                icon.style.width = map_settings.style.pin_width;
              }
              if (map_settings.style.pin_height !== '') {
                icon.style.height = map_settings.style.pin_height;
              }
              // The marker location.
              const position = new google.maps.LatLng(map_locations[j].lat, map_locations[j].lon);
              // Create the Marker (pin) object with all its information and add it to the Markers array.
              const marker = new google.maps.marker.AdvancedMarkerElement({
                map,
                position: position,
                content: icon,
                title: map_locations[j].marker_label,
                gmpClickable: true,
              });
              markers.push(marker);

              // Handle all popup functionality.
              if (map_locations[j].popup) {
                // Initialize infowindow (popup) with settings.
                const infowindow = new google.maps.InfoWindow({
                  content: marker.html,
                  ariaLabel: map_locations[j].marker_label,
                  disableAutoPan: map_settings.popup.disable_autopan,
                  minWidth: parseInt(map_settings.popup.min_width),
                  maxWidth: parseInt(map_settings.popup.max_width),
                })
                var open_event = map_settings.popup.open_event
                google.maps.event.addListener(marker, open_event,
                  (function (map, marker, map_settings) {
                    return function () {
                      if (infowindow.opened &&
                        map_settings.popup.second_click == 1 &&
                        this.getZIndex()) {
                        infowindow.close()
                        this.setZIndex(0)
                        infowindow.opened = false
                      }
                      else {
                        infowindow.setContent(this.html)
                        for (var i = 0; i < markers.length; i++) {
                          markers[i].setZIndex(0)
                        }
                        this.setZIndex(1)
                        infowindow.open({
                          anchor: marker,
                          map,
                          shouldFocus: map_settings.popup.should_focus,
                        })
                        infowindow.opened = true
                      }
                    }
                  }(map, marker, map_settings)))
              }
              bounds.extend(marker.position)
            }

            var markerCluster = {};

            // Additional functionality to enable clustering of pins.
            if (typeof map_settings.cluster != 'undefined' && map_settings.cluster.cluster_enabled) {
              // Set default height and width of cluster icon.
              let clusterIconWidth = '50';
              let clusterIconHeight = '50';
              if (map_settings.cluster.width !== '') {
                clusterIconWidth =  map_settings.cluster.width;
              }
              if (map_settings.cluster.height !== '') {
                clusterIconHeight =  map_settings.cluster.height;
              }
              // Configure renderer.
              // @see https://googlemaps.github.io/js-markerclusterer/interfaces/Renderer.html
              let renderer = {
                render: function({count, position}, stats, map) {
                  const clusterIcon2 = map_settings.cluster.pin_image;
                  // Create cluster SVG element
                  const pinSvgString = `
                  <svg width="${clusterIconWidth}" height="${clusterIconHeight}" xmlns="http://www.w3.org/2000/svg">
                      <image href="${clusterIcon2}" height="${clusterIconWidth}" width="${clusterIconHeight}" />
                  </svg>`;

                  const parser = new DOMParser();
                  const pinSvg = parser.parseFromString(pinSvgString, "image/svg+xml").documentElement;

                  // Optional: style the cluster icon
                  pinSvg.setAttribute("transform", "translate(0 28)");

                  const clusterMarkerOptions = {
                    map,
                    position: position,
                    content: pinSvg,
                    title: String(count),
                    // adjust zIndex to be above other markers
                    zIndex: Number(google.maps.Marker.MAX_ZINDEX) + count,
                  };

                  return new google.maps.marker.AdvancedMarkerElement(clusterMarkerOptions);
                }
              };
              let algorithm;
              if (map_settings.cluster.cluster_algorithm == 0) {
                // Using SuperClusterAlgorithm with a minPoints option that
                // controls how many points are required to form a cluster.
                algorithm = new markerClusterer.SuperClusterAlgorithm({
                  minPoints: parseInt(map_settings.cluster.supercluster_settings.min_points),
                  maxZoom: parseInt(map_settings.cluster.supercluster_settings.max_zoom - 1),
                  minZoom: parseInt(map_settings.cluster.supercluster_settings.min_zoom),
                  radius: parseInt(map_settings.cluster.supercluster_settings.radius),
                  extent: parseInt(map_settings.cluster.supercluster_settings.extent),
                  nodeSize: parseInt(map_settings.cluster.supercluster_settings.node_size),
                });
              } else {
                algorithm = new markerClusterer.GridAlgorithm({
                  gridSize: parseInt(map_settings.cluster.grid_algorithm_settings.grid_size),
                  maxDistance: parseInt(map_settings.cluster.grid_algorithm_settings.max_distance),
                  maxZoom: parseInt(map_settings.cluster.grid_algorithm_settings.max_zoom),
                  viewportPadding: parseInt(map_settings.cluster.grid_algorithm_settings.viewport_padding),
                });
              }
              // Pass the configs to the markerCluster constructor.
              let config = { map, markers, renderer, algorithm };
              markerCluster = new markerClusterer.MarkerClusterer(config);
            }

            var markerSpiderfier = {};

            // Additional functionality to spiderify multiple close pins.
            if (typeof map_settings.spider != 'undefined' && map_settings.spider.spider_enabled) {
              var spidericonSize = new google.maps.Size(map_settings.spider.width, map_settings.spider.height);
              var spiderConfig = {
                markersWontMove: map_settings.spider.markers_wont_move,
                markersWontHide: map_settings.spider.markers_wont_hide,
                basicFormatEvents: map_settings.spider.basic_format_events,
                keepSpiderfied: map_settings.spider.keep_spiderfied,
                nearbyDistance: map_settings.spider.nearby_distance,
                circleSpiralSwitchover: map_settings.spider.circle_spiral_switchover,
                legWeight: map_settings.spider.leg_weight,
                spiralFootSeparation: map_settings.spider.spiralFootSeparation,
                circleFootSeparation: map_settings.spider.circleFootSeparation,
                icon: {
                  url: map_settings.spider.pin_image,
                  size: spidericonSize,
                  scaledSize: spidericonSize
                }
              };
              // Init OverlappingMarkerSpiderfier with map.
              markerSpiderfier = new OverlappingMarkerSpiderfier(map, spiderConfig);
              for (var m in markers ) {
                // Add the Marker to OverlappingMarkerSpiderfier.
                markerSpiderfier.addMarker(markers[m]);
              }
              // Set original pins when spiderify.
              markerSpiderfier.addListener('spiderfy', function (markers) {
                for (var i = 0; i < markers.length; i++) {
                  markers[i].setIcon(markers[i].original_icon);
                  markers[i].setZIndex(1);
                }
              });
              // Set pin icon when unspiderfy.
              markerSpiderfier.addListener('unspiderfy', function (markers) {
                for (var i = 0; i < markers.length; i++) {
                  markers[i].setIcon(spiderConfig.icon);
                  infoBubble.close();
                }
              });
              google.maps.event.addListener(map, 'idle', function (marker) {
                // Change spiderable markers to plus sign markers
                // and subsequently any other zoom/idle.
                var spidered = markerSpiderfier.markersNearAnyOtherMarker();
                for (var i = 0; i < spidered.length; i++) {
                  // Set spidered icon when inside cluster.
                  spidered[i].setIcon(spiderConfig.icon);
                  spidered[i].setZIndex(0);
                }
              });
              // triggering drag event so that the spider pin icons get enabled again
              google.maps.event.addListener(map, 'dragend', function() {
                google.maps.event.trigger(map, 'click');
              });
            }

          }

          // Add controls that are configured for views.
          var map_view = map_container.closest('.view');
          if (map_view) {
            // Try to find the map controls.
            var custom_controls = map_view.getElementsByClassName('google-map-control');
            if (custom_controls) {
              // Add controls to the positions that are provided.
              for (var index = 0; index < custom_controls.length; index++) {
                var position = custom_controls[index].getAttribute('data-position');
                var id = custom_controls[index].getAttribute('id');
                map.controls[google.maps.ControlPosition[position]].push(document.getElementById(id));
              }
            }
          }

          // Handle centering of the map.
          if (map_settings.map_center && map_settings.map_center.center_coordinates) {
            if (!isNaN(parseInt(map_settings.map_center.center_coordinates.lat)) && !isNaN(parseInt(map_settings.map_center.center_coordinates.lon))) {
              var map_center = new google.maps.LatLng(map_settings.map_center.center_coordinates.lat, map_settings.map_center.center_coordinates.lon);
              bounds.extend(map_center);
              map.setCenter(map_center);
            }
          }
          else {
            map.setCenter(bounds.getCenter());
          }

          // Process settings of the directions plugin.
          if (typeof map_settings.directions != 'undefined' && map_settings.directions.enabled) {
            var directionsDisplay = new google.maps.DirectionsRenderer;
            var directionsService = new google.maps.DirectionsService;
            directionsDisplay.setMap(map);
            if (map_settings.directions.steps) {
              var route_details_id = 'steps-' + maps[i];
              directionsDisplay.setPanel(document.getElementById(route_details_id));
            }
            var button_id = 'find-directions-' + maps[i];

            var current_location = 'getting_directions_location-' + maps[i];
            if (document.location.protocol === 'https:') {
              document.addEventListener('geolocation_error', function(error){
                document.getElementById('directions' + maps[i]).getElementsByClassName('error-message').item(0).textContent = error.code;
              });
              document.getElementById(current_location).addEventListener('change', function () {
                var address = 'getting_directions_address-' + maps[i];
                if (this.checked) {
                  settings.user_current_location = settings.user_current_location || false;
                  var user_location = new GeolocationMarker(map);
                  if (user_location) {
                    settings.user_current_location = user_location;
                    document.getElementById(address).style.display = 'none';
                  }
                } else {
                  document.getElementById(address).style.display = 'block';
                }
              });
            } else {
              document.getElementById(current_location).parentNode.style.display = 'none';
            }
            document.getElementById(button_id).addEventListener('click', function() {
              calculateAndDisplayRoute(directionsService, directionsDisplay, maps[i], map_locations[0]);
            });
          }

          // This is needed to set the zoom after fitbounds.
          google.maps.event.addListener(map, 'zoom_changed', function () {
            var zoomChangeBoundsListener =
              google.maps.event.addListener(map, 'bounds_changed', function (event) {
                var current_zoom = this.getZoom();
                if (current_zoom > parseInt(map_settings.zoom.default) && map.initialZoom == true) {
                  // Change max/min zoom here.
                  this.setZoom(parseInt(map_settings.zoom.default) - 1);
                }
                map.initialZoom = false;
                google.maps.event.removeListener(zoomChangeBoundsListener);
              });
          });

          map.initialZoom = true;
          map.fitBounds(bounds);
          settings.initialized_styles_google_maps = settings.initialized_styles_google_maps || [];
          settings.initialized_styles_google_maps[maps[i]] = {map: map, markers: markers, cluster: markerCluster, spider: markerSpiderfier, heatmap: heatmap};
        }
      }
      // Prevents piling up generated map ids.
      settings.styled_google_map = [];
    }
  };

})(Drupal);
