<?php

namespace Drupal\styled_google_map\Plugin\Field\FieldFormatter;

use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\field\Entity\FieldConfig;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\styled_google_map\StyledGoogleMapInterface;

/**
 * Plugin implementation of the 'styled_google_map_default' formatter.
 *
 * @FieldFormatter(
 *   id = "styled_google_map_default",
 *   label = @Translation("Styled Google Map"),
 *   field_types = {
 *     "geofield"
 *   }
 * )
 */
class StyledGoogleMapDefaultFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'width' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_WIDTH,
      'height' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEIGHT,
      'gestureHandling' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_GESTURE,
      'style' => [
        'maptype' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAP_TYPE,
        'map_id' => '',
        'pin' => '',
        'pin_width' => '',
        'pin_height' => '',
      ],
      'map_center' => [
        'center_coordinates' => NULL,
      ],
      'popup' => [
        'choice' => NULL,
        'default_state' => 0,
        'second_click' => 0,
        'close_button_source' => NULL,
        'text' => NULL,
        'view_mode' => NULL,
        'open_event' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_OPEN_EVENT,
        'label' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_LABEL,
        'min_width' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MIN_WIDTH,
        'max_width' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAX_WIDTH,
        'disable_autopan' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DISABLE_AUTO_PAN,
      ],
      'zoom' => [
        'default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ZOOM,
        'max' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAX_ZOOM,
        'min' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MIN_ZOOM,
      ],
      'directions' => [
        'enabled' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DIRECTIONS,
        'type' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DIRECTIONS_TYPE,
        'steps' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DIRECTIONS_SHOW_STEPS,
      ],
      'maptypecontrol' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAP_TYPE_CONTROL,
      'scalecontrol' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SCALE_CONTROL,
      'rotatecontrol' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ROTATE_CONTROL,
      'draggable' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DRAGGABLE,
      'mobile_draggable' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MOBILE_DRAGGABLE,
      'zoomcontrol' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ZOOM_CONTROL,
      'fullscreen' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_FULLSCREEN,
      'streetviewcontrol' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_STREET_VIEW_CONTROL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);
    $default_settings = StyledGoogleMapDefaultFormatter::defaultSettings();
    // Set all available setting fields for the Styled Google Map.
    $elements['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width size'),
      '#default_value' => $this->getSetting('width'),
      '#description' => $this->t('Map width written in pixels or percentage'),
      '#required' => TRUE,
    ];
    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height size'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Map height written in pixels or percentage'),
      '#required' => TRUE,
    ];
    $elements['gestureHandling'] = [
      '#type' => 'select',
      '#title' => $this->t('Gesture handling'),
      '#description' => $this->t('This setting controls how the API handles gestures on the map. See more <a href="@href">here</a>',
        [
          '@href' => 'https://developers.google.com/maps/documentation/javascript/reference/map#MapOptions.gestureHandling',
        ]
      ),
      '#options' => [
        'cooperative' => $this->t('Scroll events with a ctrl key or ⌘ key pressed zoom the map.'),
        'greedy' => $this->t('All touch gestures and scroll events pan or zoom the map.'),
        'none' => $this->t('The map cannot be panned or zoomed by user gestures.'),
        'auto' => $this->t('(default) Gesture handling is either cooperative or greedy'),
      ],
      '#default_value' => $this->getSetting('gestureHandling'),
    ];
    $elements['style'] = [
      '#type' => 'details',
      '#title' => $this->t('Map style'),
    ];
    $style_settings = $this->getSetting('style');
    $elements['style']['maptype'] = [
      '#type' => 'select',
      '#options' => [
        'ROADMAP' => $this->t('ROADMAP'),
        'SATELLITE' => $this->t('SATELLITE'),
        'HYBRID' => $this->t('HYBRID'),
        'TERRAIN' => $this->t('TERRAIN'),
      ],
      '#title' => $this->t('Map type'),
      '#default_value' => empty($style_settings['maptype']) ? $default_settings['style']['maptype'] : $style_settings['maptype'],
      '#required' => TRUE,
    ];
    $elements['style']['map_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google map ID'),
      '#default_value' => empty($style_settings['map_id']) ? $default_settings['style']['map_id'] : $style_settings['map_id'],
      '#description' => $this->t('For information on how to create map id, visit: <a href="@link">@link</a>', [
        '@link' => 'https://developers.google.com/maps/documentation/get-map-id#create-map-ids',
      ]),
      '#required' => TRUE,
    ];
    $elements['style']['pin'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL to the marker'),
      '#default_value' => empty($style_settings['pin']) ? $default_settings['style']['pin'] : $style_settings['pin'],
      '#description' => $this->t('URL to the marker image. You can use a @wrapper for the url. Ex. @example (not working until @fixed)',
        [
          '@wrapper' => Link::fromTextAndUrl($this->t('Stream wrapper'),
            Url::fromUri('https://drupal.org/project/system_stream_wrapper'))
            ->toString(),
          '@example' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_PIN,
          '@fixed' => Link::fromTextAndUrl('https://www.drupal.org/node/1308152', Url::fromUri('https://www.drupal.org/node/1308152'))
            ->toString(),
        ]
      ),
    ];
    $elements['style']['pin_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pin width'),
      '#default_value' => empty($style_settings['pin_width']) ? '' : $style_settings['pin_width'],
      '#description' => $this->t('If you want to scale the pin to certain dimensions. If you set width, height is also required'),
    ];
    $elements['style']['pin_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pin height'),
      '#default_value' => empty($style_settings['pin_height']) ? '' : $style_settings['pin_height'],
      '#description' => $this->t('If you want to scale the pin to certain dimensions. If you set height, width is also required'),
    ];
    $map_center_settings = $this->getSetting('map_center');
    $elements['map_center'] = [
      '#type' => 'details',
      '#title' => $this->t('Centering map'),
    ];
    // Retrieve all field names from the current entity bundle.
    $field_options = ['label' => $this->t('Entity label')];
    $center_options = ['' => $this->t('Center automatically')];
    $fields = $form['#fields'];
    foreach ($fields as $field) {
      $config = FieldConfig::loadByName($form['#entity_type'], $form['#bundle'], $field);
      if (!$config) {
        continue;
      }
      $type = $config->get('field_type');
      $name = $config->get('field_name');
      $field_options[$field] = $config->getLabel();
      if ($type == 'geofield' && $this->fieldDefinition->getName() != $name) {
        $center_options[$field] = $config->getLabel();
      }
    }

    $elements['map_center']['center_coordinates'] = [
      '#type' => 'select',
      '#options' => $center_options,
      '#default_value' => empty($map_center_settings['center_coordinates']) ? $default_settings['map_center']['center_coordinates'] : $map_center_settings['center_coordinates'],
      '#description' => $this->t('To have map centered on other point than location you need to have another GeoField in your content type structure'),
    ];
    $elements['popup'] = [
      '#type' => 'details',
      '#title' => $this->t('Marker popup'),
    ];
    $popup_settings = $this->getSetting('popup');
    $elements['popup']['choice'] = [
      '#title' => $this->t('How to render popup'),
      '#type' => 'select',
      '#options' => [
        0 => $this->t('No need'),
        1 => $this->t('From Field'),
        2 => $this->t('From View mode'),
      ],
      '#default_value' => empty($popup_settings['choice']) ? $default_settings['popup']['choice'] : $popup_settings['choice'],
      '#id' => 'edit-popup-choice-field',
    ];
    // Retrieve view mode settings from the current entity bundle.
    $view_modes = \Drupal::service('entity_display.repository')
      ->getViewModeOptions($form['#entity_type']);
    $elements['popup']['view_mode'] = [
      '#title' => $this->t('Select the view mode'),
      '#type' => 'select',
      '#options' => $view_modes,
      '#default_value' => empty($popup_settings['view_mode']) ? $default_settings['popup']['view_mode'] : $popup_settings['view_mode'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['value' => 2],
        ],
      ],
    ];

    $elements['popup']['text'] = [
      '#title' => $this->t('Select field'),
      '#type' => 'select',
      '#options' => $field_options,
      '#default_value' => empty($popup_settings['text']) ? $default_settings['popup']['text'] : $popup_settings['text'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['value' => 1],
        ],
      ],
    ];
    $elements['popup']['label'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show field label'),
      '#default_value' => empty($popup_settings['label']) ? $default_settings['popup']['label'] : $popup_settings['label'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['value' => 1],
        ],
      ],
    ];
    $elements['popup']['default_state'] = [
      '#title' => $this->t('Popup default state'),
      '#type' => 'select',
      '#options' => [
        '0' => $this->t('Closed'),
        '1' => $this->t('Opened'),
      ],
      '#default_value' => empty($popup_settings['default_state']) ? $default_settings['popup']['default_state'] : $popup_settings['default_state'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['!value' => 0],
        ],
      ],
    ];
    $elements['popup']['second_click'] = [
      '#title' => $this->t('Close popup on second mouse click'),
      '#type' => 'select',
      '#options' => [
        '0' => $this->t('No'),
        '1' => $this->t('Yes'),
      ],
      '#default_value' => !isset($popup_settings['second_click']) ? $default_settings['popup']['second_click'] : $popup_settings['second_click'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['!value' => 0],
        ],
      ],
    ];
    $elements['popup']['close_button_source'] = [
      '#title' => $this->t('Popup close button image'),
      '#type' => 'textfield',
      '#default_value' => empty($popup_settings['close_button_source']) ? $default_settings['popup']['close_button_source'] : $popup_settings['close_button_source'],
      '#states' => [
        'visible' => [
          ':input[id="edit-popup-choice-field"]' => ['!value' => 0],
        ],
      ],
    ];
    $elements['popup']['open_event'] = [
      '#type' => 'select',
      '#options' => [
        'click' => $this->t('On click'),
        'mouseover' => $this->t('On hover'),
      ],
      '#title' => $this->t('Mouse event for opening popup'),
      '#default_value' => empty($popup_settings['open_event']) ? $default_settings['popup']['open_event'] : $popup_settings['open_event'],
    ];
    $elements['popup']['shadow_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Shadow style'),
      '#options' => [0, 1, 2],
      '#description' => $this->t('1: shadow behind, 2: shadow below, 0: no shadow'),
      '#default_value' => empty($popup_settings['shadow_style']) ? $default_settings['popup']['shadow_style'] : $popup_settings['shadow_style'],
    ];
    $elements['popup']['min_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Min width'),
      '#field_suffix' => 'px (or auto)',
      '#default_value' => empty($popup_settings['min_width']) ? $default_settings['popup']['min_width'] : $popup_settings['min_width'],
    ];
    $elements['popup']['max_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Max width'),
      '#field_suffix' => 'px (or auto)',
      '#default_value' => empty($popup_settings['max_width']) ? $default_settings['popup']['max_width'] : $popup_settings['max_width'],
    ];
    $elements['popup']['disable_autopan'] = [
      '#type' => 'select',
      '#title' => $this->t('Auto pan'),
      '#options' => [TRUE => $this->t('Yes'), FALSE => $this->t('No')],
      '#description' => $this->t('Automatically center the pin on click'),
      '#default_value' => empty($popup_settings['disable_autopan']) ? $default_settings['popup']['disable_autopan'] : $popup_settings['disable_autopan'],
    ];
    $elements['zoom'] = [
      '#type' => 'details',
      '#title' => $this->t('Zoom'),
    ];
    $zoom_settings = $this->getSetting('zoom');
    $elements['zoom']['default'] = [
      '#type' => 'select',
      '#options' => range(1, 23),
      '#title' => $this->t('Default zoom level'),
      '#default_value' => empty($zoom_settings['default']) ? $default_settings['zoom']['default'] : $zoom_settings['default'],
      '#description' => $this->t('Should be between the Min and Max zoom level.'),
      '#required' => TRUE,
    ];
    $elements['zoom']['max'] = [
      '#type' => 'select',
      '#options' => range(1, 23),
      '#title' => $this->t('Max zoom level'),
      '#default_value' => empty($zoom_settings['max']) ? $default_settings['zoom']['max'] : $zoom_settings['max'],
      '#description' => $this->t('Should be greater then the Min zoom level.'),
      '#required' => TRUE,
    ];
    $elements['zoom']['min'] = [
      '#type' => 'select',
      '#options' => range(1, 23),
      '#title' => $this->t('Min zoom level'),
      '#default_value' => empty($zoom_settings['min']) ? $default_settings['zoom']['min'] : $zoom_settings['min'],
      '#description' => $this->t('Should be smaller then the Max zoom level.'),
      '#required' => TRUE,
    ];
    $elements['directions'] = [
      '#type' => 'details',
      '#title' => $this->t('Getting directions'),
    ];
    $directions_settings = $this->getSetting('directions');
    $elements['directions']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable'),
      '#default_value' => empty($directions_settings['enabled']) ? $default_settings['directions']['enabled'] : $directions_settings['enabled'],
      '#description' => $this->t('If you want to show "Get directions" block on the map'),
    ];
    $elements['directions']['type'] = [
      '#type' => 'select',
      '#options' => [
        'SELECTABLE' => $this->t('Allow user to choose'),
        'DRIVING' => $this->t('Driving (indicates standard driving directions using the road network)'),
        'BICYCLING' => $this->t('Bicycling (requests bicycling directions via bicycle paths & preferred streets)'),
        'TRANSIT' => $this->t('Public transport'),
        'WALKING' => $this->t('Walking'),
      ],
      '#title' => $this->t('Travel mode'),
      '#default_value' => empty($directions_settings['type']) ? $default_settings['directions']['type'] : $directions_settings['type'],
      '#description' => $this->t('Read more about <a href="@url" target="_blank">travel mode</a>.', ['@url' => 'https://developers.google.com/maps/documentation/javascript/directions#TravelModes']),
      '#required' => TRUE,
    ];
    $elements['directions']['steps'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show detailed steps in a separate window'),
      '#default_value' => empty($directions_settings['steps']) ? $default_settings['directions']['steps'] : $directions_settings['steps'],
      '#description' => $this->t('When enabled will show detailed steps of the route in a separate window.'),
    ];
    $elements['maptypecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Map Type control'),
      '#default_value' => $this->getSetting('maptypecontrol'),
    ];
    $elements['scalecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable scale control'),
      '#default_value' => $this->getSetting('scalecontrol'),
    ];
    $elements['rotatecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate control'),
      '#default_value' => $this->getSetting('rotatecontrol'),
    ];
    $elements['draggable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable dragging'),
      '#default_value' => $this->getSetting('draggable'),
    ];
    $elements['mobile_draggable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable mobile dragging'),
      '#description' => $this->t('Sometimes when the map covers big part of touch device screen draggable feature can cause inability to scroll the page'),
      '#default_value' => $this->getSetting('mobile_draggable'),
    ];
    $elements['streetviewcontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable street view control'),
      '#default_value' => $this->getSetting('streetviewcontrol'),
    ];
    $elements['zoomcontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable zoom control'),
      '#default_value' => $this->getSetting('zoomcontrol'),
    ];
    $elements['fullscreen'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable full screen control'),
      '#default_value' => $this->getSetting('fullscreen'),
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Width: <b>%width</b><br />Height: <b>%height</b>',
      [
        '%width' => $this->getSetting('width'),
        '%height' => $this->getSetting('height'),
      ]);
    $style_settings = $this->getSetting('style');
    if ($style_settings['map_id']) {
      $summary[] = $this->t('<br />MapId for map style: <b>Custom</b>');
    }
    if ($style_settings['pin']) {
      $summary[] = $this->t('<br />Pin style: <b>%pin</b>', ['%pin' => $style_settings['pin']]);
    }
    $summary[] = $this->t('<br />Map type: <b>%maptype</b>', ['%maptype' => $style_settings['maptype']]);
    if ($style_settings['pin']) {
      $summary[] = $this->t('<br />Pin location: <b>%pin</b>', ['%pin' => $style_settings['pin']]);
    }
    $popup_settings = $this->getSetting('popup');
    if ($popup_settings['choice'] == 1) {
      $summary[] = $this->t('<br />Popup shows field <b>%field</b>', ['%field' => $popup_settings['text']]);
      $readable = [FALSE => $this->t('without'), TRUE => $this->t('with')];
      $summary[] = $this->t('<b>%label</b> label', ['%label' => $readable[$popup_settings['label']]]);
    }
    if ($popup_settings['choice'] == 2) {
      $summary[] = $this->t('<br />Popup shows view mode <b>%viewmode</b>', ['%viewmode' => $popup_settings['view_mode']]);
    }
    $zoom_settings = $this->getSetting('zoom');
    $summary[] = $this->t('<br />Default zoom: <b>%zoom</b>', ['%zoom' => $zoom_settings['default']]);
    $summary[] = $this->t('<br />Maximum zoom: <b>%maxzoom</b>', ['%maxzoom' => $zoom_settings['max']]);
    $summary[] = $this->t('<br />Minimum zoom: <b>%minzoom</b>', ['%minzoom' => $zoom_settings['min']]);
    $gesture_handling = $this->getSetting('gestureHandling');
    $summary[] = $this->t('<br />Gesture handling:<b>%mode</b>', ['%mode' => $gesture_handling]);
    $directions_settings = $this->getSetting('directions');
    if ($directions_settings['enabled']) {
      $summary[] = $this->t('<br />Getting directions option is enabled with such settings:');
      $summary[] = $this->t('<br />Directions travel mode: <b>%mode</b>', ['%mode' => $directions_settings['type']]);
      $summary[] = $this->t('<br />Show detailed steps: <b>%steps</b>', ['%steps' => $directions_settings['steps'] ? $this->t('Yes') : $this->t('No')]);
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $parent_entity = $items->getEntity();
    $elements = [];
    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        '#theme' => 'styled_google_map',
        '#location' => $item->getValue(),
        '#settings' => $this->getSettings(),
        '#entity' => $parent_entity,
        '#gid' => uniqid(),
      ];
    }
    return $elements;
  }

}
