<?php

namespace Drupal\styled_google_map\Plugin\views\style;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\styled_google_map\StyledGoogleMapInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Drupal\views\Render\ViewsRenderPipelineMarkup;

/**
 * Views area StyledGoogleMapStyle handler.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "styled_google_map",
 *   title = @Translation("Styled Google Map"),
 *   help = @Translation("Displays geofield values on the Google Map with
 *   styles."), theme = "views_view_table", display_types = {"normal"}
 * )
 */
class StyledGoogleMapStyle extends StylePluginBase {
  /**
   * Does the style plugin for itself support to add fields to it's output.
   *
   * @var bool
   */
  protected $usesFields = TRUE;

  /**
   * Does the style plugin allows to use style plugins.
   *
   * @var bool
   */
  protected $usesRowPlugin = FALSE;

  /**
   * Does the style plugin support custom css class for the rows.
   *
   * @var bool
   */
  protected $usesRowClass = FALSE;

  /**
   * Should field labels be enabled by default.
   *
   * @var bool
   */
  protected $defaultFieldLabels = FALSE;

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    // Default main options.
    $options['main'] = [
      'contains' => [
        'styled_google_map_id' => ['default' => ''],
        'styled_google_map_gesture_handling' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_GESTURE],
        'styled_google_map_view_height' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_WIDTH],
        'styled_google_map_view_width' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEIGHT],
        'styled_google_map_view_pin_width' => ['default' => ''],
        'styled_google_map_view_pin_height' => ['default' => ''],
        'styled_google_map_view_zoom_default' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ZOOM],
        'styled_google_map_view_zoom_max' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAX_ZOOM],
        'styled_google_map_view_zoom_min' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MIN_ZOOM],
        'styled_google_map_view_maptype' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAP_TYPE],
        'styled_google_map_view_maptypecontrol' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAP_TYPE_CONTROL],
        'styled_google_map_view_scalecontrol' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SCALE_CONTROL],
        'styled_google_map_view_rotatecontrol' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ROTATE_CONTROL],
        'styled_google_map_view_draggable' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DRAGGABLE],
        'styled_google_map_view_mobile_draggable' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MOBILE_DRAGGABLE],
        'styled_google_map_view_zoomcontrol' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_ZOOM_CONTROL],
        'styled_google_map_view_fullscreen' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_FULLSCREEN],
        'styled_google_map_view_streetviewcontrol' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_STREET_VIEW_CONTROL],
        'styled_google_map_default_map_center' => ['default' => ['lat' => 0, 'lon' => 90]],
      ],
    ];
    $options['marker_label_settings'] = [
      'contains' => [
        'color' => ['default' => 'black'],
        'fontFamily' => ['default' => 'Arial'],
        'fontSize' => ['default' => '14px'],
        'fontWeight' => ['default' => 'normal'],
      ],
    ];
    // Default popup options.
    $options['popup'] = [
      'contains' => [
        'open_event' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_OPEN_EVENT],
        'second_click' => ['default' => 0],
        'styled_google_map_view_disable_auto_pan' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_DISABLE_AUTO_PAN],
        'styled_google_map_view_min_width' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MIN_WIDTH],
        'styled_google_map_view_max_width' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_MAX_WIDTH],
        'styled_google_map_view_should_focus' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SHOULD_FOCUS],
      ],
    ];
    // Default cluster settings.
    $options['cluster_settings'] = [
      'contains' => [
        'cluster_enabled' => ['default' => 0],
        'cluster_algorithm' => ['default' => 0],
        'pin_image' => ['default' => ''],
        'height' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_HEIGHT],
        'width' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_WIDTH],
      ],
    ];
    // Default cluster settings for supercluster algorithm.
    $options['supercluster_settings'] = [
      'contains' => [
        'min_zoom' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_MIN_ZOOM],
        'max_zoom' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_MAX_ZOOM],
        'radius' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_RADIUS],
        'extent' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_EXTENT],
        'node_size' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_NODE_SIZE],
        'min_points' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_SUPERCLUSTER_MIN_POINTS],
      ],
    ];
    // Default cluster settings for grid algorithm.
    $options['grid_algorithm_settings'] = [
      'contains' => [
        'grid_size' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_GRID_GRID_SIZE],
        'max_distance' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_GRID_MAX_DISTANCE],
        'max_zoom' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_GRID_MAX_ZOOM],
        'viewport_padding' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_CLUSTER_GRID_VIEWPORT_PADDING],
      ],
    ];
    // Default spider settings.
    $options['spider_settings'] = [
      'contains' => [
        'spider_enabled' => ['default' => 0],
        'pin_image' => ['default' => ''],
        'height' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_HEIGHT],
        'width' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_WIDTH],
        'markers_wont_move' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_MARKERS_WONT_MOVE],
        'markers_wont_hide' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_MARKERS_WONT_HIDE],
        'basic_format_events' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_BASIC_FORMAT_EVENTS],
        'keep_spiderfied' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_KEEP_SPIDERFIED],
        'nearby_distance' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_NEARBY_DISTANCE],
        'circle_spiral_switchover' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_CIRCLE_SPIRAL_SWITCHOVER],
        'leg_weight' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_LEG_WEIGHT],
        'circleFootSeparation' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_CIRCLE_FOOT_SAPARATION],
        'spiralFootSeparation' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_SPIDERFIER_SPIRAL_FOOT_SAPARATION],
      ],
    ];
    // Default heat map settings.
    $options['heatmap_settings'] = [
      'contains' => [
        'heatmap_enabled' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_ENABLED],
        'dissipating' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_DISSIPATING],
        'gradient' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_GRADIENT],
        'maxIntensity' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_MAX_INTENSITY],
        'opacity' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_OPACITY],
        'radius' => ['default' => StyledGoogleMapInterface::STYLED_GOOGLE_MAP_DEFAULT_HEATMAP_RADIUS],
      ],
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $handlers = $this->displayHandler->getHandlers('field');
    $data_source_options = ['' => $this->t('-- Choose the field --')];
    $pin_source_options = ['' => $this->t('-- Choose the field --')];
    $source_options = ['' => $this->t('-- Choose the field --')];
    /** @var \Drupal\views\Plugin\views\ViewsHandlerInterface $handle */
    foreach ($handlers as $key => $handle) {
      // Get all location sources.
      if (!empty($handle->options['type']) && $handle->options['type'] == 'geofield_default') {
        $data_source_options[$key] = $handle->adminLabel();
      }
      // Get all pin sources.
      if (!empty($handle->options['type']) && $handle->options['type'] == 'image') {
        $pin_source_options[$key] = $handle->adminLabel();
      }
      // Get all popup sources.
      $source_options[$key] = $handle->adminLabel();
    }

    $form['data_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains geodata?'),
      '#description' => $this->t('Needs to be a geofield.'),
      '#required' => TRUE,
      '#options' => $data_source_options,
      '#default_value' => $this->options['data_source'] ? $this->options['data_source'] : NULL,
    ];
    $form['pin_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains the pin image?'),
      '#description' => $this->t('Needs to be an image field.'),
      '#options' => $pin_source_options,
      '#default_value' => $this->options['pin_source'] ? $this->options['pin_source'] : NULL,
    ];
    $form['default_pin_source'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default pin image'),
      '#default_value' => $this->options['default_pin_source'],
      '#description' => $this->t('Also you can have a default pin image for all the locations'),
    ];
    $form['active_pin_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains the active pin image?'),
      '#description' => $this->t('Needs to be an image field. This will only be applicable when pin source has some field'),
      '#options' => $pin_source_options,
      '#default_value' => $this->options['active_pin_source'] ? $this->options['active_pin_source'] : NULL,
    ];
    $form['popup_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains the popup text?'),
      '#description' => $this->t('Can be a field or rendered entity field.'),
      '#options' => $source_options,
      '#default_value' => $this->options['popup_source'] ? $this->options['popup_source'] : NULL,
    ];
    $form['marker_label'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains the marker label?'),
      '#description' => $this->t('Can be a field or rendered entity field.'),
      '#options' => $source_options,
      '#default_value' => $this->options['marker_label'] ? $this->options['marker_label'] : NULL,
    ];
    $form['category_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains the category?'),
      '#description' => $this->t('This will be used to have a class wrapper around the bubble to allow different styling per category.'),
      '#options' => $source_options,
      '#default_value' => $this->options['category_source'] ? $this->options['category_source'] : NULL,
    ];
    $form['main'] = [
      '#type' => 'details',
      '#title' => $this->t('Map Options'),
    ];
    $form['main']['styled_google_map_view_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#size' => '30',
      '#description' => $this->t('This field determines how width the styled Google map'),
      '#default_value' => $this->options['main']['styled_google_map_view_width'],
    ];
    $form['main']['styled_google_map_view_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#size' => '30',
      '#description' => $this->t('This field determines the height of the styled Google map'),
      '#default_value' => $this->options['main']['styled_google_map_view_height'],
    ];
    $form['main']['styled_google_map_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google map ID'),
      '#description' => $this->t('For information on how to create map id, visit: <a href="@link">@link</a>', [
        '@link' => 'https://developers.google.com/maps/documentation/get-map-id#create-map-ids',
      ]),
      '#required' => TRUE,
      '#default_value' => $this->options['main']['styled_google_map_id'],
    ];
    $form['main']['styled_google_map_view_pin_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Pin Height'),
      '#size' => '30',
      '#description' => $this->t('If you want to scale the pin to certain dimensions. If you set width, height is also required'),
      '#default_value' => $this->options['main']['styled_google_map_view_pin_height'],
    ];
    $form['main']['styled_google_map_view_pin_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Pin Width'),
      '#size' => '30',
      '#description' => $this->t('If you want to scale the pin to certain dimensions. If you set height, width is also required'),
      '#default_value' => $this->options['main']['styled_google_map_view_pin_width'],
    ];
    $form['main']['styled_google_map_view_maptype'] = [
      '#type' => 'select',
      '#options' => [
        'ROADMAP' => $this->t('ROADMAP'),
        'SATELLITE' => $this->t('SATELLITE'),
        'HYBRID' => $this->t('HYBRID'),
        'TERRAIN' => $this->t('TERRAIN'),
      ],
      '#title' => $this->t('Map type'),
      '#default_value' => $this->options['main']['styled_google_map_view_maptype'],
      '#required' => TRUE,
    ];
    $form['main']['styled_google_map_gesture_handling'] = [
      '#type' => 'select',
      '#title' => $this->t('Gesture handling'),
      '#description' => $this->t('This setting controls how the API handles gestures on the map. See more <a href="@href">here</a>',
        [
          '@href' => 'https://developers.google.com/maps/documentation/javascript/reference/map#MapOptions.gestureHandling',
        ]
      ),
      '#options' => [
        'cooperative' => $this->t('Scroll events with a ctrl key or ⌘ key pressed zoom the map.'),
        'greedy' => $this->t('All touch gestures and scroll events pan or zoom the map.'),
        'none' => $this->t('The map cannot be panned or zoomed by user gestures.'),
        'auto' => $this->t('(default) Gesture handling is either cooperative or greedy'),
      ],
      '#default_value' => $this->options['main']['styled_google_map_gesture_handling'],
    ];
    $form['main']['styled_google_map_view_zoom_default'] = [
      '#type' => 'select',
      '#options' => range(1, 35),
      '#title' => $this->t('Default zoom level'),
      '#default_value' => $this->options['main']['styled_google_map_view_zoom_default'],
      '#description' => $this->t('Should be between the Min and Max zoom level.
        This will generally not working as fitbounds will try to fit all pins on the map.'),
      '#required' => TRUE,
    ];
    $form['main']['styled_google_map_view_zoom_max'] = [
      '#type' => 'select',
      '#options' => range(1, 35),
      '#title' => $this->t('Max zoom level'),
      '#default_value' => $this->options['main']['styled_google_map_view_zoom_max'],
      '#description' => $this->t('Should be greater then the Min zoom level.'),
      '#required' => TRUE,
    ];
    $form['main']['styled_google_map_view_zoom_min'] = [
      '#type' => 'select',
      '#options' => range(1, 35),
      '#title' => $this->t('Min zoom level'),
      '#default_value' => $this->options['main']['styled_google_map_view_zoom_min'],
      '#description' => $this->t('Should be smaller then the Max zoom level.'),
      '#required' => TRUE,
    ];
    $form['main']['styled_google_map_view_maptypecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Map Type control'),
      '#default_value' => $this->options['main']['styled_google_map_view_maptypecontrol'],
    ];
    $form['main']['styled_google_map_view_scalecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable scale control'),
      '#default_value' => $this->options['main']['styled_google_map_view_scalecontrol'],
    ];
    $form['main']['styled_google_map_view_rotatecontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate control'),
      '#default_value' => $this->options['main']['styled_google_map_view_rotatecontrol'],
    ];
    $form['main']['styled_google_map_view_draggable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable dragging'),
      '#default_value' => $this->options['main']['styled_google_map_view_draggable'],
    ];
    $form['main']['styled_google_map_view_mobile_draggable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable mobile dragging'),
      '#description' => $this->t('Sometimes when the map covers big part of touch device screen draggable feature can cause inability to scroll the page'),
      '#default_value' => $this->options['main']['styled_google_map_view_mobile_draggable'],
    ];
    $form['main']['styled_google_map_view_streetviewcontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable street view control'),
      '#default_value' => $this->options['main']['styled_google_map_view_streetviewcontrol'],
    ];
    $form['main']['styled_google_map_view_zoomcontrol'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable zoom control'),
      '#default_value' => $this->options['main']['styled_google_map_view_zoomcontrol'],
    ];
    $form['main']['styled_google_map_view_fullscreen'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable full screen control'),
      '#default_value' => $this->options['main']['styled_google_map_view_fullscreen'],
    ];
    $form['main']['styled_google_map_default_map_center'] = [
      '#type' => 'geofield_latlon',
      '#title' => $this->t('Default map center coordinates'),
      '#default_value' => $this->options['main']['styled_google_map_default_map_center'],
    ];
    $form['marker_label_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Marker label settings'),
      '#states' => [
        'invisible' => [
          'select[name="style_options[marker_label]"]' => ['value' => ''],
        ],
      ],
    ];
    $form['marker_label_settings']['color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Colour'),
      '#description' => $this->t('The color of the label text. Default color is black.'),
      '#default_value' => $this->options['marker_label_settings']['color'],
    ];
    $form['marker_label_settings']['fontFamily'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Font family'),
      '#description' => $this->t('The font family of the label text (equivalent to the CSS font-family property).'),
      '#default_value' => $this->options['marker_label_settings']['fontFamily'],
    ];
    $form['marker_label_settings']['fontSize'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Font size'),
      '#description' => $this->t('The font size of the label text (equivalent to the CSS font-size property). Default size is 14px.'),
      '#default_value' => $this->options['marker_label_settings']['fontSize'],
    ];
    $form['marker_label_settings']['fontWeight'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Font weight'),
      '#description' => $this->t('The font weight of the label text (equivalent to the CSS font-weight property).'),
      '#default_value' => $this->options['marker_label_settings']['fontWeight'],
    ];
    $form['cluster_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Cluster settings'),
    ];
    $form['cluster_settings']['cluster_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable clustering'),
      '#default_value' => $this->options['cluster_settings']['cluster_enabled'],
    ];
    $form['cluster_settings']['pin_image'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cluster pin image'),
      '#default_value' => $this->options['cluster_settings']['pin_image'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cluster_settings']['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cluster image height'),
      '#default_value' => $this->options['cluster_settings']['height'],
      '#field_suffix' => $this->t('pixels'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cluster_settings']['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cluster image width'),
      '#default_value' => $this->options['cluster_settings']['width'],
      '#field_suffix' => $this->t('pixels'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cluster_settings']['cluster_algorithm'] = [
      '#type' => 'radios',
      '#title' => $this->t('Clustering algorithm'),
      '#default_value' => $this->options['cluster_settings']['cluster_algorithm'],
      '#options' => [0 => $this->t('Supercluster'), 1 => $this->t('GridAlgorithm')],
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cluster_settings']['supercluster_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Supercluster settings'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_algorithm]"]' => ['value' => 0],
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['cluster_settings']['grid_algorithm_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('GridAlgorithm settings'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[cluster_settings][cluster_algorithm]"]' => ['value' => 1],
          ':input[name="style_options[cluster_settings][cluster_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    // Settings form for Supercluster algorithm.
    $form['cluster_settings']['supercluster_settings']['min_points'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum cluster size'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['min_points'],
      '#description' => $this->t('The minimum number of pins to be grouped in a cluster.'),
    ];
    $form['cluster_settings']['supercluster_settings']['min_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum zoom'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['min_zoom'],
      '#description' => $this->t('The minimum zoom level at which clusters are generated.'),
    ];
    $form['cluster_settings']['supercluster_settings']['max_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum cluster size'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['max_zoom'],
      '#description' => $this->t('The maximum zoom level at which clusters are generated.'),
    ];
    $form['cluster_settings']['supercluster_settings']['radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Cluster radius'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['radius'],
      '#field_suffix' => $this->t('pixels'),
      '#description' => $this->t('The Cluster radius, in pixels.'),
    ];
    $form['cluster_settings']['supercluster_settings']['extent'] = [
      '#type' => 'number',
      '#title' => $this->t('Tile extent'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['extent'],
      '#description' => $this->t('(Tiles) Tile extent. Radius is calculated relative to this value.'),
    ];
    $form['cluster_settings']['supercluster_settings']['node_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Node size'),
      '#default_value' => $this->options['cluster_settings']['supercluster_settings']['node_size'],
      '#description' => $this->t('The Size of the KD-tree leaf node. Affects performance.'),
    ];
    // Settings form for GridAlgorithm algorithm.
    $form['cluster_settings']['grid_algorithm_settings']['grid_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Grid size'),
      '#default_value' => $this->options['cluster_settings']['grid_algorithm_settings']['grid_size'],
    ];
    $form['cluster_settings']['grid_algorithm_settings']['max_distance'] = [
      '#type' => 'number',
      '#title' => $this->t('Max distance'),
      '#default_value' => $this->options['cluster_settings']['grid_algorithm_settings']['max_distance'],
      '#description' => $this->t('Max distance between cluster center and point in meters.'),
    ];
    $form['cluster_settings']['grid_algorithm_settings']['max_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Max zoom'),
      '#default_value' => $this->options['cluster_settings']['grid_algorithm_settings']['max_zoom'],
      '#description' => $this->t('Markers are not clustered at maxZoom and above.'),
    ];
    $form['cluster_settings']['grid_algorithm_settings']['viewport_padding'] = [
      '#type' => 'number',
      '#title' => $this->t('Viewport padding'),
      '#default_value' => $this->options['cluster_settings']['grid_algorithm_settings']['viewport_padding'],
      '#description' => $this->t('The number of pixels to extend beyond the viewport bounds when filtering markers prior to clustering.'),
    ];
    $form['spider_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Spider settings'),
    ];
    $form['spider_settings']['spider_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Spider'),
      '#default_value' => $this->options['spider_settings']['spider_enabled'],
    ];
    $form['spider_settings']['markers_wont_move'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Do not move markers'),
      '#default_value' => $this->options['spider_settings']['markers_wont_move'],
      '#description' => $this->t('Spedir option from config: markersWontMove'),
    ];
    $form['spider_settings']['markers_wont_hide'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Do not hide markers'),
      '#default_value' => $this->options['spider_settings']['markers_wont_hide'],
      '#description' => $this->t('Spedir option from config: markersWontHide'),
    ];
    $form['spider_settings']['basic_format_events'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Listen to basic format events'),
      '#default_value' => $this->options['spider_settings']['basic_format_events'],
      '#description' => $this->t('Spedir option from config: basicFormatEvents'),
    ];
    $form['spider_settings']['pin_image'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spider pin image'),
      '#default_value' => $this->options['spider_settings']['pin_image'],
    ];
    $form['spider_settings']['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spider image height'),
      '#default_value' => $this->options['spider_settings']['height'],
      '#suffix' => $this->t('pixels'),
    ];
    $form['spider_settings']['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spider image width'),
      '#default_value' => $this->options['spider_settings']['width'],
      '#suffix' => $this->t('pixels'),
    ];
    $form['spider_settings']['keep_spiderfied'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keep markers spiderfied when clicked'),
      '#default_value' => $this->options['spider_settings']['keep_spiderfied'],
    ];
    $form['spider_settings']['nearby_distance'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The pixel radius within which a marker is considered to be overlapping a clicked marker'),
      '#default_value' => $this->options['spider_settings']['nearby_distance'],
    ];
    $form['spider_settings']['circle_spiral_switchover'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The lowest number of markers that will be fanned out into a spiral instead of a circle'),
      '#description' => $this->t('Set this to 0 to always get spirals, or "Infinity" for all circles.'),
      '#default_value' => $this->options['spider_settings']['circle_spiral_switchover'],
    ];
    $form['spider_settings']['leg_weight'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The thickness of the lines joining spiderfied markers to their original locations'),
      '#default_value' => $this->options['spider_settings']['leg_weight'],
    ];
    $form['spider_settings']['circleFootSeparation'] = [
      '#type' => 'number',
      '#title' => $this->t('Circle foot separation to maintain distance between the markers'),
      '#default_value' => $this->options['spider_settings']['circleFootSeparation'],
    ];
    $form['spider_settings']['spiralFootSeparation'] = [
      '#type' => 'number',
      '#title' => $this->t('Spiral Foot Separation to maintain distance between the markers'),
      '#default_value' => $this->options['spider_settings']['spiralFootSeparation'],
    ];
    $form['heatmap_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Heat map settings'),
    ];
    $form['heatmap_settings']['heatmap_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable heat map layer'),
      '#description' => $this->t('When this feature is enabled you can not add markers anymore'),
      '#default_value' => $this->options['heatmap_settings']['heatmap_enabled'],
    ];
    $form['heatmap_settings']['data_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Which field contains heat map weight?'),
      '#description' => $this->t('Location data is taken from main view settings. Read more about weight <a href="@url">here</a>', ['@url' => 'https://developers.google.com/maps/documentation/javascript/heatmaplayer']),
      '#options' => $source_options,
      '#default_value' => $this->options['heatmap_settings']['data_source'] ? $this->options['heatmap_settings']['data_source'] : NULL,
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['heatmap_settings']['dissipating'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Dissipating'),
      '#description' => $this->t('Specifies whether heatmaps dissipate on zoom. By default, the radius of influence of a data point is specified by the radius option only. When dissipating is disabled, the radius option is interpreted as a radius at zoom level 0.'),
      '#default_value' => $this->options['heatmap_settings']['dissipating'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['heatmap_settings']['gradient'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Gradient'),
      '#description' => $this->t('The color gradient of the heatmap, specified as an array of CSS color strings. All CSS3 colors are supported except for extended named colors. Write one colour per line.'),
      '#default_value' => $this->options['heatmap_settings']['gradient'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['heatmap_settings']['opacity'] = [
      '#type' => 'number',
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
      '#title' => $this->t('Opacity'),
      '#description' => $this->t('The opacity of the heatmap, expressed as a number between 0 and 1. Defaults to 0.6.'),
      '#default_value' => $this->options['heatmap_settings']['opacity'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['heatmap_settings']['radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Radius'),
      '#description' => $this->t('The radius of influence for each data point, in pixels.'),
      '#default_value' => $this->options['heatmap_settings']['radius'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['heatmap_settings']['maxIntensity'] = [
      '#type' => 'number',
      '#title' => $this->t('Max intensity'),
      '#description' => $this->t('The maximum intensity of the heatmap. By default, heatmap colors are dynamically scaled according to the greatest concentration of points at any particular pixel on the map. This property allows you to specify a fixed maximum.'),
      '#default_value' => $this->options['heatmap_settings']['maxIntensity'],
      '#states' => [
        'visible' => [
          ':input[name="style_options[heatmap_settings][heatmap_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['popup'] = [
      '#type' => 'details',
      '#title' => $this->t('Popup Styling'),
      '#description' => $this->t('All settings for the popup exposed by the library. If you want more flexibility in your the styling of the popup. You can use the CSS defined classes'),
    ];
    $form['popup']['second_click'] = [
      '#title' => $this->t('Close popup on second mouse click'),
      '#type' => 'select',
      '#options' => [
        '0' => $this->t('No'),
        '1' => $this->t('Yes'),
      ],
      '#default_value' => $this->options['popup']['second_click'],
    ];
    $form['popup']['close_button_source'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close button image'),
      '#default_value' => $this->options['popup']['close_button_source'],
    ];
    $form['popup']['open_event'] = [
      '#type' => 'select',
      '#options' => [
        'click' => $this->t('On click'),
        'mouseover' => $this->t('On hover'),
      ],
      '#title' => $this->t('Mouse event for opening popup'),
      '#default_value' => $this->options['popup']['open_event'],
    ];
    $form['popup']['styled_google_map_view_min_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Min width'),
      '#field_suffix' => 'px (or auto)',
      '#default_value' => $this->options['popup']['styled_google_map_view_min_width'],
    ];
    $form['popup']['styled_google_map_view_max_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Max width'),
      '#field_suffix' => 'px (or auto)',
      '#default_value' => $this->options['popup']['styled_google_map_view_max_width'],
    ];
    $form['popup']['styled_google_map_view_disable_auto_pan'] = [
      '#type' => 'select',
      '#title' => $this->t('Auto pan'),
      '#options' => [TRUE => $this->t('Yes'), FALSE => $this->t('No')],
      '#description' => $this->t('Automatically center the pin on click'),
      '#default_value' => $this->options['popup']['styled_google_map_view_disable_auto_pan'],
    ];
    $form['popup']['styled_google_map_view_should_focus'] = [
      '#type' => 'select',
      '#title' => $this->t('Should focus'),
      '#options' => [TRUE => $this->t('Yes'), FALSE => $this->t('No')],
      '#description' => $this->t('Whether or not focus should be moved inside the popup'),
      '#default_value' => $this->options['popup']['styled_google_map_view_should_focus'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    // We check if the views result are empty, or if the settings of this area
    // force showing this area even if the view is empty.
    if (!empty($this->view->live_preview)) {
      $output['preview'] = [
        '#markup' => '<p>' . $this->t('This is a preview of styled google map plugin. No map is displayed.') . '</p>',
      ];
      $output['maps'] = [
        '#markup' => '<p>' . $this->t('This map has @num points', ['@num' => count($this->view->result)]) . '</p>',
      ];
      return $output;
    }

    $locations = [];
    $heat_map_coords = [];
    // Get all geofield locations.
    foreach ($this->view->result as $row_index => $row) {
      $sourceHandlerOutput = [];
      // Render all fields first, so they are available for token replacement.
      /** @var \Drupal\views\Plugin\views\field\Field $handler */
      foreach ($this->displayHandler->getHandlers('field') as $field => $handler) {
        $handler->view->row_index = $row_index;
        $output = $handler->advancedRender($row);
        $placeholders = $handler->postRender($row, $output);
        $output = ViewsRenderPipelineMarkup::create(str_replace(array_keys($placeholders), array_values($placeholders), $output));
        $sourceHandlerOutput[$field] = $output;
      }
      $location = [];
      if (!empty($this->options['data_source']) && !empty($sourceHandlerOutput[$this->options['data_source']])) {
        // Add geofield data.
        try {
          $geom = \geoPHP::load($sourceHandlerOutput[$this->options['data_source']]);
          if (!empty($geom)) {
            /** @var \Geometry $centroid */
            $centroid = $geom->getCentroid();
            $point = [];
            $point['lon'] = $centroid->getX();
            $point['lat'] = $centroid->getY();
            $location = $location + $point;
            if ($this->options['heatmap_settings']['heatmap_enabled']) {
              $heat_map_coords[] = $point + ['weight' => 0];
            }
          }
          else {
            continue;
          }
        }
        catch (\Exception $e) {
          continue;
        }

        if (!empty($this->options['default_pin_source'])) {
          $location['pin'] = styled_google_map_file_create_url($this->options['default_pin_source']);
          if (!empty($this->options['main']['styled_google_map_view_active_pin'])) {
            $location['active_pin'] = styled_google_map_file_create_url($this->options['main']['styled_google_map_view_active_pin']);
          }
        }
        // Add pin image url.
        if (!empty($this->options['pin_source']) && !empty($sourceHandlerOutput[$this->options['pin_source']])) {
          /** @var \Drupal\views\Plugin\views\field\FieldHandlerInterface $sourceHandler */
          $sourceHandler = $this->displayHandler->getHandler('field', $this->options['pin_source']);
          $fileTargetId = $sourceHandler->render($row);
          if ($fileTargetId instanceof ViewsRenderPipelineMarkup) {
            $image = File::load($fileTargetId->__toString());
            $location['pin'] = styled_google_map_file_create_url($image->getFileUri());
            // Add the active pin image.
            if (!empty($this->options['active_pin_source']) && !empty($sourceHandlerOutput[$this->options['active_pin_source']])) {
              /** @var \Drupal\views\Plugin\views\field\FieldHandlerInterface $sourceHandler */
              $sourceHandler = $this->displayHandler->getHandler('field', $this->options['active_pin_source']);
              $fileTargetId = $sourceHandler->render($row);
              if ($fileTargetId instanceof ViewsRenderPipelineMarkup) {
                $image = File::load($fileTargetId->__toString());
                $location['active_pin'] = styled_google_map_file_create_url($image->getFileUri());
              }
            }
            elseif (empty($this->options['main']['styled_google_map_view_active_pin'])) {
              $location['active_pin'] = styled_google_map_file_create_url($image->getFileUri());
            }
            else {
              $location['active_pin'] = styled_google_map_file_create_url($this->options['main']['styled_google_map_view_active_pin']);
            }
          }
          elseif (!empty($this->options['default_pin_source'])) {
            $location['pin'] = styled_google_map_file_create_url($this->options['default_pin_source']);
          }
        }
        // Add marker Label.
        if (!empty($this->options['marker_label']) && isset($sourceHandlerOutput[$this->options['marker_label']])) {
          $markerLabelRenderArray = ['#markup' => $sourceHandlerOutput[$this->options['marker_label']]];
          $marker = \Drupal::service('renderer')->renderPlain($markerLabelRenderArray);
          $marker = strip_tags($marker);
          $location = $location + [
              'marker_label' => [
                'text' => $marker,
                'color' => $this->options['marker_label_settings']['color'],
                'fontFamily' => $this->options['marker_label_settings']['fontFamily'],
                'fontSize' => $this->options['marker_label_settings']['fontSize'],
                'fontWeight' => $this->options['marker_label_settings']['fontWeight'],
              ],
            ];
        }
        // Add pin popup html.
        if (!empty($this->options['popup_source']) && !empty($sourceHandlerOutput[$this->options['popup_source']])) {
          $popupRenderArray = ['#markup' => $sourceHandlerOutput[$this->options['popup_source']]];
          $location = $location + [
              'popup' => \Drupal::service('renderer')->renderPlain($popupRenderArray),
            ];
        }
        // Add category.
        if (!empty($this->options['category_source']) && !empty($sourceHandlerOutput[$this->options['category_source']])) {
          $categoryRenderArray = ['#markup' => $sourceHandlerOutput[$this->options['category_source']]];
          $category = \Drupal::service('renderer')->renderPlain($categoryRenderArray);
          $location = $location + [
              'category' => Html::cleanCssIdentifier($category),
            ];
        }
      }
      if ($location) {
        $locations[] = $location;
      }
      // Gather heatmap weights.
      if (!empty($this->options['heatmap_settings']['data_source']) && !empty($sourceHandlerOutput[$this->options['heatmap_settings']['data_source']])) {
        $weightRenderArray = ['#markup' => $sourceHandlerOutput[$this->options['heatmap_settings']['data_source']]];
        $weight = \Drupal::service('renderer')->renderPlain($weightRenderArray);
        $heat_map_coords[$row_index]['weight'] = $weight;
      }
    }
    // Add custom settings.
    $cluster = [];

    if ($this->options['cluster_settings']['cluster_enabled']) {
      $cluster_pin_image = styled_google_map_file_create_url($this->options['cluster_settings']['pin_image']);
      $cluster = ['pin_image' => $cluster_pin_image] + $this->options['cluster_settings'];
    }

    // Spider settings.
    $spider = [];
    if ($this->options['spider_settings']['spider_enabled']) {
      $spider_pin_image = styled_google_map_file_create_url($this->options['spider_settings']['pin_image']);
      $spider = ['pin_image' => $spider_pin_image] + $this->options['spider_settings'];
    }
    // Heat map settings.
    $heat_map = [];
    if ($this->options['heatmap_settings']['heatmap_enabled'] && !empty($heat_map_coords)) {
      $heat_map = ['data' => $heat_map_coords] + $this->options['heatmap_settings'];
      if (!empty($heat_map['gradient'])) {
        $heat_map['gradient'] = explode("\r\n", $heat_map['gradient']);
      }
    }
    // @todo sanitize all options.
    $rand = rand();
    $map_id = $this->view->dom_id . $rand;
    $map_settings = [
      'id' => 'map_' . $map_id,
      'locations' => $locations,
      'settings' => [
        'gestureHandling' => $this->options['main']['styled_google_map_gesture_handling'],
        'height' => $this->options['main']['styled_google_map_view_height'],
        'width' => $this->options['main']['styled_google_map_view_width'],
        'maptypecontrol' => $this->options['main']['styled_google_map_view_maptypecontrol'],
        'scalecontrol' => $this->options['main']['styled_google_map_view_scalecontrol'],
        'rotatecontrol' => $this->options['main']['styled_google_map_view_rotatecontrol'],
        'draggable' => $this->options['main']['styled_google_map_view_draggable'],
        'mobile_draggable' => $this->options['main']['styled_google_map_view_mobile_draggable'],
        'streetviewcontrol' => $this->options['main']['styled_google_map_view_streetviewcontrol'],
        'style' => [
          'maptype' => $this->options['main']['styled_google_map_view_maptype'],
          'map_id' => $this->options['main']['styled_google_map_id'],
          'pin_width' => $this->options['main']['styled_google_map_view_pin_width'],
          'pin_height' => $this->options['main']['styled_google_map_view_pin_height'],
        ],
        'zoom' => [
          'default' => $this->options['main']['styled_google_map_view_zoom_default'],
          'max' => $this->options['main']['styled_google_map_view_zoom_max'],
          'min' => $this->options['main']['styled_google_map_view_zoom_min'],
        ],
        'zoomcontrol' => $this->options['main']['styled_google_map_view_zoomcontrol'],
        'fullscreen' => $this->options['main']['styled_google_map_view_fullscreen'],
        'marker_label' => [
          'color' => $this->options['marker_label_settings']['color'],
          'fontFamily' => $this->options['marker_label_settings']['fontFamily'],
          'fontSize' => $this->options['marker_label_settings']['fontSize'],
          'fontWeight' => $this->options['marker_label_settings']['fontWeight'],
        ],
        'popup' => [
          'open_event' => $this->options['popup']['open_event'] ? $this->options['popup']['open_event'] : 'click',
          'second_click' => $this->options['popup']['second_click'] ? $this->options['popup']['second_click'] : 0,
          'disable_autopan' => $this->options['popup']['styled_google_map_view_disable_auto_pan'] ? TRUE : FALSE,
          'min_width' => $this->options['popup']['styled_google_map_view_min_width'],
          'max_width' => $this->options['popup']['styled_google_map_view_max_width'],
          'should_focus' => $this->options['popup']['styled_google_map_view_should_focus'],
        ],
      ],
    ];
    // If cluster feature is enabled.
    if (!empty($cluster)) {
      $map_settings['settings']['cluster'] = $cluster;
    }
    // If spiderfier feature is enabled.
    if (!empty($spider)) {
      $map_settings['settings']['spider'] = $spider;
    }
    // If heat map feature is enabled.
    if (!empty($heat_map)) {
      $map_settings['settings']['heat_map'] = $heat_map;
    }

    // Check if the custom map center option is enabled.
    if (!empty($this->options['main']['styled_google_map_default_map_center'])) {
      $map_settings['settings']['map_center'] = [
        'center_coordinates' => [
          'lat' => $this->options['main']['styled_google_map_default_map_center']['lat'],
          'lon' => $this->options['main']['styled_google_map_default_map_center']['lon'],
        ],
      ];
    }

    // Allow other modules to change the styled_google_map settings.
    $alter_vars = [
      'map_settings' => $map_settings,
      'context' => [
        'view' => $this->view,
        'options' => $this->options,
      ],
    ];
    \Drupal::moduleHandler()->alter('styled_google_map_views_style', $alter_vars);
    $map_settings = $alter_vars['map_settings'];

    // Prepare the output of the view style.
    $output = [];
    $output['#attached']['drupalSettings']['styled_google_map'] = ['map_' . $map_id => 'map_' . $map_id];
    $output['#attached']['drupalSettings']['maps'] = ['idmap_' . $map_id => $map_settings];
    // Output a div placeholder for the Styled Google Map.
    $output['styled_google_map']['#markup'] = '<div class="styled_map" id="map_' . $map_id . '"></div>';
    // Attach the Styled Google Map javascript file.
    $output['#attached']['library'][] = 'styled_google_map/styled-google-map';
    if (!empty($cluster)) {
      $output['#attached']['library'][] = 'styled_google_map/google-map-clusters';
    }
    if (!empty($spider)) {
      $output['#attached']['library'][] = 'styled_google_map/spiderfier';
    }
    if (!empty($heat_map)) {
      $output['#attached']['library'][] = 'styled_google_map/heatmap';
    }
    return $output;
  }

  /**
   * Always render the map even when there are no markers available.
   *
   * @return bool
   *   Returns whether the view should be rendered with no results.
   */
  public function evenEmpty() {
    return TRUE;
  }

}
