<?php

namespace Drupal\sucuri_cache\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\sucuri_cache\SucuriCachePurger;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Purges Sucuri cache for selected entities.
 *
 * @Action(
 *   id = "purge_sucuri_cache",
 *   action_label = @Translation("Purge Sucuri Cache")
 * )
 */
class PurgeSucuriCacheAction extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The SucuriCachePurger service.
   *
   * @var \Drupal\sucuri_cache\SucuriCachePurger
   */
  protected $cachePurger;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Creates a new PurgeSucuriCacheAction plugin instance.
   *
   * @param \Drupal\sucuri_cache\SucuriCachePurger $cache_purger
   *   SucuriCachePurger.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(SucuriCachePurger $cache_purger, MessengerInterface $messenger, array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->cachePurger = $cache_purger;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('sucuri_cache.purger'),
      $container->get('messenger'),
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(?EntityInterface $entity = NULL) {
    if ($entity && $this->cachePurger->purgeCacheForEntity($entity)) {
      $this->messenger->addMessage(
        $this->t('Sucuri cache has been cleared for %title.',
          ['%title' => $entity->label()]
        )
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    /** @var \Drupal\Core\Entity\EntityInterface $object */
    $result = $object->access('update', $account, TRUE);
    return $return_as_object ? $result : $result->isAllowed();
  }

}
