<?php

namespace Drupal\sucuri_cache\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sucuri_cache\SucuriCachePurger;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Clears sucuri caches for a concrete page.
 *
 * @internal
 */
class PurgeEntityCacheForm extends ConfirmFormBase implements ContainerInjectionInterface {

  /**
   * The entity to purge cache for.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $entity;

  /**
   * The Sucuri cache purger service.
   *
   * @var \Drupal\sucuri_cache\SucuriCachePurger
   */
  protected $sucuriCachePurger;

  /**
   * Constructs a new PurgeEntityCacheForm.
   *
   * @param \Drupal\sucuri_cache\SucuriCachePurger $sucuri_cache_purger
   *   The Sucuri cache purger service.
   */
  public function __construct(SucuriCachePurger $sucuri_cache_purger) {
    $this->sucuriCachePurger = $sucuri_cache_purger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sucuri_cache.purger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sucuri_cache__purge_entity_cache_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $node = NULL) {
    // @todo allow not only node.
    $this->entity = $node;
    return parent::buildForm($form, $form_state);
  }

  /**
   * Clears the caches.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $result = $this->sucuriCachePurger->purgeCacheForEntity($this->entity);
    
    if ($result) {
      $this->messenger()->addStatus($this->t('Sucuri cache cleared successfully for %title.', [
        '%title' => $this->entity->label(),
      ]));
    }
    else {
      $this->messenger()->addError($this->t('Failed to clear Sucuri cache for %title. Please check the logs for more details.', [
        '%title' => $this->entity->label(),
      ]));
    }
    
    // Redirect to the content overview page.
    $form_state->setRedirect('system.admin_content');
  }

  /**
   * Confirmation action.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   A string of question.
   */
  public function getQuestion() {
    return $this->t('Do you want to purge sucuri caches for %title?', [
      '%title' => $this->entity->label(),
    ]);
  }

  /**
   * Return URL for entity.
   *
   * @return \Drupal\Core\Url
   *   Url object.
   */
  public function getCancelUrl() {
    return $this->entity->toUrl('canonical');
  }

}
