<?php

namespace Drupal\sucuri_cache;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service class responsible for cache purging.
 */
final class SucuriCachePurger {

  use StringTranslationTrait;

  const ACTION_CLEAR_CACHE = 'clear_cache';

  /**
   * API Client service.
   *
   * @var \Drupal\sucuri_cache\WafApiClient
   */
  private $apiClient;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  private $messenger;

  /**
   * Constructs an SucuriCachePurger object.
   *
   * @param \Drupal\sucuri_cache\WafApiClient $api_client
   *   API Client.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger Interface.
   */
  public function __construct(WafApiClient $api_client, MessengerInterface $messenger) {
    $this->apiClient = $api_client;
    $this->messenger = $messenger;
  }

  /**
   * Cache clear for a specific entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Entity object whose cache need to clear.
   *
   * @return bool
   *   Return Boolean.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function purgeCacheForEntity(EntityInterface $entity): bool {
    return $this->purgeCache([
      'file' => $entity->toUrl()->toString(),
    ]);
  }

  /**
   * Purge cache for all pages in the site.
   *
   * @return bool
   *   Return Boolean.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function purgeAllCache(): bool {
    return $this->purgeCache();
  }

  /**
   * Performs cache purge operation via API.
   *
   * @param array $extra_query_params
   *   Additional query parameters for the API request.
   *
   * @return bool
   *   TRUE if purge was successful, FALSE otherwise.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  private function purgeCache(array $extra_query_params = []): bool {
    $res = $this->apiClient->requestAction(self::ACTION_CLEAR_CACHE, $extra_query_params);

    // @todo not sure about about success code here.
    if ($res && $res->getStatusCode() === 200) {
      // Success.
      return TRUE;
    }
    return FALSE;
  }

}
