<?php

namespace Drupal\sucuri_cache;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;

/**
 * WAF API interaction service class.
 */
final class WafApiClient {

  use StringTranslationTrait;

  /**
   * HTTP-client service.
   *
   * @var \GuzzleHttp\Client
   */
  private $httpClient;

  /**
   * Manager service.
   *
   * @var \Drupal\sucuri_cache\SucuriCacheManager
   */
  private $manager;

  /**
   * Cached API credentials.
   *
   * @var array|null
   */
  private $credentials = NULL;

  /**
   * Class constructor.
   */
  public function __construct(Client $http_client, SucuriCacheManager $manager) {
    $this->httpClient = $http_client;
    $this->manager = $manager;
  }

  /**
   * Gets cached API credentials.
   *
   * @return array
   *   Array containing api_endpoint, api_key, and api_secret.
   */
  private function getCredentials(): array {
    if ($this->credentials === NULL) {
      $this->credentials = $this->manager->getApiCredentials();
    }
    return $this->credentials;
  }

  /**
   * Sends an action request to API endpoint.
   *
   * @param string $action
   *   Action code to be performed.
   * @param array $extra_query_params
   *   Other query parameter for request.
   *
   * @return false|\Psr\Http\Message\ResponseInterface|null
   *   Return status of request.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   *   Exception in API request.
   */
  public function requestAction(string $action, array $extra_query_params = []) {
    $credentials = $this->getCredentials();
    $endpoint = $credentials['api_endpoint'];
    $api_key = $credentials['api_key'];
    $api_secret = $credentials['api_secret'];

    if (empty($endpoint) || empty($api_key) || empty($api_secret)) {
      $this->manager->logError('Cannot call WAF API, credentials are not provided.');
      return FALSE;
    }

    $headers = [];
    $query = [
      'k' => $api_key,
      's' => $api_secret,
      'a' => $action,
    ];

    $options = [
      'http_errors' => TRUE,
      'connect_timeout' => 10,
      'read_timeout' => 10,
      'timeout' => 10,
      'verify' => FALSE,
      // @todo No cert check?
      'headers' => $headers,
      'query' => $query + $extra_query_params,
    ];

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      
      // Log successful API response.
      $response_body = (string) $response->getBody();
      \Drupal::logger('sucuri_cache')->info('API @action response: @response', [
        '@action' => $action,
        '@response' => $response_body,
      ]);
      
      return $response;
    }
    catch (ConnectException $e) {
      // Connection timeout.
      $this->manager->logError('Connection timed out: @msg', ['@msg' => $e->getMessage()]);
      return FALSE;
    }
    catch (RequestException $e) {
      // Bad response.
      $response = $e->getResponse();
      $this->manager->logError('Request failed with code @code: @body', [
        '@code' => $response->getStatusCode(),
        '@body' => (string) $response->getBody(),
      ]);
      return $response;
    }
    catch (\Exception $e) {
      // Unknown error.
      $this->manager->logError('Unexpected error: @msg', ['@msg' => $e->getMessage()]);
      return FALSE;
    }
  }

}
