<?php

namespace Drupal\sudoku\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\sudoku\Service\SudokuGenerator;
use Drupal\sudoku\Service\SudokuSolver;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Sudoku AJAX actions and listing.
 */
class SudokuController extends ControllerBase {

  /**
   * Logger channel for logging messages.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Sudoku puzzle generator service.
   *
   * @var \Drupal\sudoku\Service\SudokuGenerator
   */
  protected $generator;

  /**
   * Sudoku puzzle solver service.
   *
   * @var \Drupal\sudoku\Service\SudokuSolver
   */
  protected $solver;

  public function __construct(
    LoggerInterface $logger,
    SudokuGenerator $generator,
    SudokuSolver $solver,
  ) {
    $this->logger = $logger;
    $this->generator = $generator;
    $this->solver = $solver;
  }

  /**
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The instantiated controller.
   */
  public static function create(ContainerInterface $container): static {
    // Get the named logger channel 'sudoku'.
    $logger = $container->get('logger.factory')->get('sudoku');
    $generator = $container->get('sudoku.generator');
    $solver = $container->get('sudoku.solver');
    return new static($logger, $generator, $solver);
  }

  /**
   * Admin listing of Sudoku entities.
   *
   * @return array
   *   Render array.
   */
  public function collection(): array {
    $build = [];
    $storage = $this->entityTypeManager()->getStorage('sudoku');

    // Load all entities, ignoring access for simplicity.
    $ids = $storage->getQuery()->accessCheck(FALSE)->execute();
    $entities = $storage->loadMultiple($ids);

    // Build table rows.
    $rows = [];
    foreach ($entities as $e) {
      $link = Link::fromTextAndUrl($this->t('View'), $e->toUrl())->toString();
      $rows[] = [
        'data' => [
          $e->id(),
          $e->label(),
          $link,
        ],
      ];
    }

    // Render the table.
    $build['table'] = [
      '#type' => 'table',
      '#header' => [$this->t('ID'), $this->t('Title'), $this->t('Operations')],
      '#rows' => $rows,
    ];

    return $build;
  }

  /**
   * AJAX solver endpoint. Expects JSON body with key 'grid' => 9x9 array.
   *
   * Returns JSON with 'solution' => solved grid, or 'error' => message.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function solveAjax(Request $request): JsonResponse {
    $data = json_decode($request->getContent(), TRUE) ?: [];
    $grid = $data['grid'] ?? $data['puzzle'] ?? NULL;
    if (!$grid) {
      return new JsonResponse(['error' => 'No grid provided.'], 400);
    }

    try {
      $solution = $this->solver->solve($grid);
      if ($solution === FALSE) {
        return new JsonResponse(['error' => 'Unable to solve puzzle.'], 200);
      }
      return new JsonResponse(['solution' => $solution], 200);
    }
    catch (\Throwable $e) {
      $this->logger->error('Sudoku solveAjax exception: @m', [
        '@m' => $e->getMessage(),
      ]);
      return new JsonResponse([
        'error' => 'Server error during solve. Check logs.',
      ], 500);
    }
  }

  /**
   * AJAX puzzle generator endpoint.
   *
   * Generates a new Sudoku puzzle of a given difficulty. Expects JSON body with
   * key 'difficulty' => 'easy'|'medium'|'hard'.
   *
   * Returns JSON with 'puzzle' => generated grid, or 'error' => message.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function generateAjax(Request $request): JsonResponse {
    try {
      // Start timing the generation.
      $start = microtime(TRUE);
      $data = json_decode($request->getContent(), TRUE) ?: [];
      $difficulty = $data['difficulty'] ?? 'medium';

      // Validate difficulty.
      $puzzle = $this->generator->generate($difficulty);

      // Log duration and check for failure.
      $duration = microtime(TRUE) - $start;
      if (!$puzzle) {
        $this->logger->error(
          'Sudoku generation returned no puzzle for difficulty @d (took @t s).',
          ['@d' => $difficulty, '@t' => $duration]
        );
        return new JsonResponse(['error' => 'Generation failed.'], 500);
      }

      // Log success.
      $this->logger->info('Generated sudoku (difficulty: @d) in @t seconds.', [
        '@d' => $difficulty,
        '@t' => round($duration, 3),
      ]);
      return new JsonResponse(['puzzle' => $puzzle], 200);
    }
    catch (\Throwable $e) {
      // Log exception details.
      $this->logger->error('Sudoku generateAjax exception: @m', [
        '@m' => $e->getMessage(),
      ]);
      $this->logger->error('Stack trace: @t', [
        '@t' => $e->getTraceAsString(),
      ]);
      return new JsonResponse([
        'error' => 'Server error during puzzle generation. Check logs.',
      ], 500);
    }
  }

}
