<?php

namespace Drupal\sudoku\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the Sudoku entity.
 *
 * @ContentEntityType(
 *   id = "sudoku",
 *   label = @Translation("Sudoku"),
 *   handlers = {
 *     "list_builder" = "Drupal\sudoku\SudokuListBuilder",
 *     "form" = {
 *       "add" = "Drupal\sudoku\Form\SudokuForm",
 *       "edit" = "Drupal\sudoku\Form\SudokuForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm"
 *     },
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder"
 *   },
 *   base_table = "sudoku",
 *   admin_permission = "administer sudoku",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "title",
 *     "uuid" = "uuid"
 *   },
 *   links = {
 *     "canonical" = "/sudoku/{sudoku}",
 *     "add-form" = "/sudoku/add",
 *     "edit-form" = "/sudoku/{sudoku}/edit",
 *     "delete-form" = "/sudoku/{sudoku}/delete",
 *     "collection" = "/admin/content/sudoku"
 *   }
 * )
 */
class Sudoku extends ContentEntityBase {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $ent_type) {
    $fields = parent::baseFieldDefinitions($ent_type);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Title'))
      ->setRequired(TRUE)
      ->setDefaultValue('Sudoku Puzzle');

    // Store the puzzle as JSON: 9x9 matrix, 0 for blank.
    $fields['puzzle'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Puzzle JSON'))
      ->setDescription(t('JSON encoded puzzle grid (9x9).'))
      ->setDefaultValue('[]');

    // Optional stored solution (JSON).
    $fields['solution'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Solution JSON'))
      ->setDescription(t('JSON encoded solution grid (9x9).'))
      ->setDefaultValue('[]');

    // Created/changed.
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback('Drupal\sudoku\Entity\Sudoku::getCurrentUserId');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'));

    return $fields;
  }

  /**
   * Default value callback for the uid field.
   *
   * Returns an array of default values (the current user id).
   *
   * Drupal expects callback that returns an array for entity_reference default.
   */
  public static function getCurrentUserId() {
    // Return an array of values where the array value is an associative array
    // with 'target_id' keyed values or just an array of user IDs depending on
    // Drupal's expectations; returning [\Drupal::currentUser()->id()] works
    // for the simple case used by most entity reference defaults.
    return [\Drupal::currentUser()->id()];
  }

}
