<?php

namespace Drupal\sudoku\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sudoku\Service\SudokuSolver;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for Sudoku entity add/edit.
 */
class SudokuForm extends ContentEntityForm {
  /**
   * The Sudoku solver service.
   *
   * @var \Drupal\sudoku\Service\SudokuSolver
   */
  protected SudokuSolver $sudokuSolver;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   *
   * Use create() to inject services from the container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   */
  public static function create(ContainerInterface $container): static {
    // Let parent create the form object (it sets up entity and other props).
    $instance = parent::create($container);

    // Inject services we need.
    $instance->sudokuSolver = $container->get('sudoku.solver');
    // Get a named logger channel for 'sudoku'.
    $instance->logger = $container->get('logger.factory')->get('sudoku');

    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * Build the form, including the 9x9 Sudoku grid.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    // Start with the parent form.
    $form = parent::buildForm($form, $form_state);
    $entity = $this->entity;

    // Decode the stored puzzle JSON, or initialize an empty grid.
    $puzzle_json = $entity->get('puzzle')->value ?: '[]';
    $puzzle = json_decode($puzzle_json, TRUE)
      ?: array_fill(0, 9, array_fill(0, 9, 0));

    // Ensure we have a 9x9 grid.
    for ($r = 0; $r < 9; $r++) {
      if (!isset($puzzle[$r]) || !is_array($puzzle[$r])) {
        $puzzle[$r] = array_fill(0, 9, 0);
      }
      else {
        for ($c = 0; $c < 9; $c++) {
          if (!isset($puzzle[$r][$c])) {
            $puzzle[$r][$c] = 0;
          }
        }
      }
    }

    // Build the 9x9 grid of textfields.
    $form['sudoku_grid'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['sudoku-grid-wrapper'],
      ],
      '#tree' => TRUE,
    ];

    // Table to hold the grid.
    $form['sudoku_grid']['grid'] = [
      '#type' => 'table',
      '#attributes' => [
        'class' => ['sudoku-grid'],
      ],
      '#header' => [],
      '#tree' => TRUE,
    ];

    // Build table rows and cells.
    for ($r = 0; $r < 9; $r++) {
      $row = [];
      for ($c = 0; $c < 9; $c++) {
        $value = $puzzle[$r][$c] ?: '';

        // Build a predictable id to match the JS selectors.
        $cell_id = 'edit-grid-' . $r . '-' . $c;

        // Each cell is a textfield.
        $row['c' . $c] = [
          '#type' => 'textfield',
          '#size' => 1,
          '#attributes' => [
            'id' => $cell_id,
            'maxlength' => 1,
            'class' => ['sudoku-cell', "r$r", "c$c"],
            'data-row' => $r,
            'data-col' => $c,
          ],
          '#default_value' => $value,
        ];
      }

      // Add the row to the table.
      $form['sudoku_grid']['grid'][$r] = $row;
    }

    // Hidden field to store the puzzle as JSON for easier JS handling.
    $form['puzzle_json'] = [
      '#type' => 'hidden',
      '#default_value' => json_encode($puzzle),
      '#name' => 'puzzle_json',
      '#attributes' => [
        'class' => ['sudoku-puzzle-json'],
        'id' => 'edit-puzzle-json',
      ],
    ];

    // Action buttons container.
    $form['actions']['solve_ajax'] = [
      '#type' => 'button',
      '#value' => $this->t('Solve (AJAX)'),
      '#attributes' => [
        'class' => ['button', 'button--primary', 'sudoku-solve'],
      ],
    ];

    // Difficulty select (give a predictable id).
    $form['actions']['generate_random'] = [
      '#type' => 'select',
      '#id' => 'edit-difficulty',
      '#options' => [
        'easy' => $this->t('Easy'),
        'medium' => $this->t('Medium'),
        'hard' => $this->t('Hard'),
      ],
      '#default_value' => 'medium',
    ];

    // Generate button — must be a non-submitting button.
    $form['actions']['generate_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate'),
      '#attributes' => [
        'class' => ['button', 'sudoku-generate'],
        'type' => 'button',
      ],
    ];

    // Attach our library for JS/CSS.
    $form['#attached']['library'][] = 'sudoku/sudoku.admin';

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * Validate the Sudoku grid input.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    // Try to get the grid the JS stores in the hidden field, then fallback.
    $values = $this->extractGridFromInput($form_state);

    // Validate each cell.
    for ($r = 0; $r < 9; $r++) {
      for ($c = 0; $c < 9; $c++) {
        $v = $values[$r][$c] ?? NULL;

        // Treat 0, empty string, and NULL as blank cells (valid).
        if ($v === '' || $v === NULL || $v === 0 || $v === '0') {
          continue;
        }

        // Must be an integer 1..9.
        if (!ctype_digit((string) $v) || ((int) $v) < 1 || ((int) $v) > 9) {
          $form_state->setErrorByName(
            "sudoku_grid][grid][{$r}][{$c}",
            $this->t(
              'Grid values must be 1-9 or blank. Error at row @r col @c.', [
                '@r' => $r + 1,
                '@c' => $c + 1,
              ]
            )
          );
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * Extract the 9x9 grid from form input.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   A 9x9 array of integers (0 for blank).
   */
  protected function extractGridFromInput(FormStateInterface $form_state) {
    // 1) Prefer structured FAPI values if present.
    $sudoku_grid = $form_state->getValue('sudoku_grid');
    if (
      is_array($sudoku_grid) &&
      isset($sudoku_grid['grid']) &&
      is_array($sudoku_grid['grid'])
    ) {
      $grid_input = $sudoku_grid['grid'];
      $grid = array_fill(0, 9, array_fill(0, 9, 0));
      foreach ($grid_input as $r => $row) {
        for ($c = 0; $c < 9; $c++) {
          $key = 'c' . $c;
          $val = $row[$key] ?? $row[$c] ?? NULL;
          $grid[$r][$c] = ($val === '' || $val === NULL)
            ? 0 : (is_numeric($val) ? (int) $val : 0);
        }
      }
      return $grid;
    }

    // 2) Next, check raw user input names like grid[0][c1]=2 (older/flat form).
    $user_input = $form_state->getUserInput();
    if (isset($user_input['grid']) && is_array($user_input['grid'])) {
      $raw = $user_input['grid'];
      $grid = array_fill(0, 9, array_fill(0, 9, 0));
      foreach ($raw as $r => $row) {
        // $row is an array with keys like 'c0', 'c1'
        for ($c = 0; $c < 9; $c++) {
          $key = 'c' . $c;
          $val = $row[$key] ?? $row[$c] ?? NULL;
          $grid[$r][$c] = ($val === '' || $val === NULL)
            ? 0 : (is_numeric($val) ? (int) $val : 0);
        }
      }
      return $grid;
    }

    // 3) Fallback: JSON in hidden input written by JS.
    $puzzle_json = $form_state->getValue('puzzle_json');
    if (!empty($puzzle_json)) {
      $decoded = @json_decode($puzzle_json, TRUE);
      if (is_array($decoded) && count($decoded) === 9) {
        for ($r = 0; $r < 9; $r++) {
          for ($c = 0; $c < 9; $c++) {
            $decoded_val = $decoded[$r][$c];
            $decoded[$r][$c] = isset($decoded_val) && $decoded_val !== ''
              ? (int) $decoded_val : 0;
          }
        }
        return $decoded;
      }
    }

    // 4) Last-resort: construct an empty grid.
    return array_fill(0, 9, array_fill(0, 9, 0));
  }

  /**
   * {@inheritdoc}
   *
   * Save the entity, including computing the solution if possible.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Current form state.
   *
   * @return int
   *   The save status constant.
   */
  public function save(array $form, FormStateInterface $form_state) {
    // Prefer structured form values (manual edits).
    // Mirrors extractGridFromInput's priority.
    $grid = $this->extractGridFromInput($form_state);

    // Convert to JSON for storage.
    $puzzle_json = json_encode($grid);

    // Validate basic shape.
    $puzzle_arr = @json_decode($puzzle_json, TRUE);
    if (!is_array($puzzle_arr) || count($puzzle_arr) !== 9) {
      $this->logger->error('Invalid puzzle JSON on save; using empty grid.');
      $puzzle_arr = array_fill(0, 9, array_fill(0, 9, 0));
      $puzzle_json = json_encode($puzzle_arr);
    }

    // Save to entity.
    $this->entity->set('puzzle', $puzzle_json);

    // Attempt to compute a solution using injected solver (best-effort).
    try {
      if (
        isset($this->sudokuSolver) &&
        is_callable([$this->sudokuSolver, 'solve'])
      ) {
        $solution = $this->sudokuSolver->solve($puzzle_arr);
        if ($solution !== FALSE) {
          $this->entity->set('solution', json_encode($solution));
        }
      }
    }
    catch (\Throwable $e) {
      $this->logger->error('Solver threw exception on save: @m', [
        '@m' => $e->getMessage(),
      ]);
    }

    // Call parent save to handle the rest.
    $status = parent::save($form, $form_state);

    // Force redirect to the canonical view of the newly saved entity.
    $form_state->setRedirect('entity.sudoku.canonical', ['sudoku' => $this->entity->id()]);

    return $status;
  }

}
