<?php

namespace Drupal\sudoku\Service;

/**
 * Sudoku generator service.
 *
 * Strategy:
 *  - Generate a fully filled valid grid using randomized backtracking.
 *  - Remove numbers according to difficulty while ensuring uniqueness (or near-
 *    unique).
 */
class SudokuGenerator {

  /**
   * The Sudoku solver service.
   *
   * @var \Drupal\sudoku\Service\SudokuSolver
   */
  protected $solver;

  public function __construct(SudokuSolver $solver) {
    $this->solver = $solver;
  }

  /**
   * Generate puzzle for difficulty: easy|medium|hard.
   *
   * Returns 9x9 puzzle array with blanks as 0 and also includes solution as
   * second element.
   *
   * @param string $difficulty
   *   Difficulty level: easy, medium, hard.
   *
   * @return bool|array
   *   Array with puzzle/solution keys on success, FALSE on failure.
   */
  public function generate(string $difficulty = 'medium'): bool|array {
    $full = $this->generateFullGrid();
    if ($full === FALSE) {
      return FALSE;
    }

    // Map difficulty to number of clues.
    $clues_map = [
      'easy' => 40,
      'medium' => 32,
      'hard' => 26,
    ];
    $clues = $clues_map[$difficulty] ?? 32;

    // Generate list of all coordinates.
    $coords = [];
    for ($r = 0; $r < 9; $r++) {
      for ($c = 0; $c < 9; $c++) {
        $coords[] = [$r, $c];
      }
    }

    // Randomize order.
    shuffle($coords);

    // Start with full grid.
    $puzzle = $full;

    // Remove until desired clues remain (81 - removals = clues).
    $removals = 81 - $clues;
    foreach ($coords as $coord) {
      if ($removals <= 0) {
        break;
      }
      // Remove and test.
      [$r, $c] = $coord;
      $backup = $puzzle[$r][$c];
      $puzzle[$r][$c] = 0;

      // Check uniqueness: if more than 1 solution, revert removal.
      $copy = $puzzle;
      $count = $this->solver->countSolutions($copy, 2);
      if ($count !== 1) {
        // Revert.
        $puzzle[$r][$c] = $backup;
      }
      else {
        // Confirm removal.
        $removals--;
      }
    }

    // Return puzzle and solution.
    return ['puzzle' => $puzzle, 'solution' => $full];
  }

  /**
   * Generate a fully filled valid Sudoku grid.
   *
   * @return bool|array
   *   Fully filled 9x9 grid on success, FALSE on failure.
   */
  protected function generateFullGrid(): bool|array {
    // Initialize empty grid.
    $grid = array_fill(0, 9, array_fill(0, 9, 0));

    // Fill grid.
    if ($this->fillGrid($grid)) {
      // Success.
      return $grid;
    }

    // Failure.
    return FALSE;
  }

  /**
   * Recursively fill grid using backtracking.
   *
   * @param array $grid
   *   Reference to 9x9 grid array.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  protected function fillGrid(array &$grid): bool {
    // Find next empty cell.
    for ($r = 0; $r < 9; $r++) {
      for ($c = 0; $c < 9; $c++) {
        if ($grid[$r][$c] == 0) {

          // Cell is empty - try numbers 1-9 in random order.
          $numbers = range(1, 9);
          shuffle($numbers);

          // Try each number.
          foreach ($numbers as $num) {
            // Check if safe.
            if ($this->isSafe($grid, $r, $c, $num)) {
              // Place number and recurse.
              $grid[$r][$c] = $num;
              // Recurse.
              if ($this->fillGrid($grid)) {
                return TRUE;
              }
              // Backtrack.
              $grid[$r][$c] = 0;
            }
          }

          return FALSE;
        }
      }
    }
    // No empties.
    return TRUE;
  }

  /**
   * Check if number can be placed at a position without violating Sudoku rules.
   *
   * @param array $grid
   *   Grid array (9x9).
   * @param int $row
   *   Row index.
   * @param int $col
   *   Column index.
   * @param int $num
   *   Number to place (1-9).
   *
   * @return bool
   *   TRUE if safe, FALSE otherwise.
   */
  protected function isSafe(
    array $grid,
    int $row,
    int $col,
    int $num,
  ): bool {
    // Row:
    for ($c = 0; $c < 9; $c++) {
      if ($grid[$row][$c] == $num) {
        return FALSE;
      }
    }

    // Column:
    for ($r = 0; $r < 9; $r++) {
      if ($grid[$r][$col] == $num) {
        return FALSE;
      }
    }

    // 3x3 box:
    $sr = $row - $row % 3;
    $sc = $col - $col % 3;
    for ($r = $sr; $r < $sr + 3; $r++) {
      for ($c = $sc; $c < $sc + 3; $c++) {
        if ($grid[$r][$c] == $num) {
          // Conflict.
          return FALSE;
        }
      }
    }

    // Safe.
    return TRUE;
  }

}
