<?php

namespace Drupal\sudoku\Service;

/**
 * Sudoku solver service using backtracking.
 */
class SudokuSolver {

  /**
   * Attempt to solve the 9x9 grid. Returns solved 9x9 array or FALSE.
   *
   * @param array $grid
   *   Sudoku grid (9x9 array with integers, 0 being blank).
   *
   * @return bool|array
   *   Solved grid or FALSE if no solution.
   */
  public function solve(array $grid): bool|array {
    // Validate shape.
    if (!$this->validShape($grid)) {
      return FALSE;
    }

    $grid_copy = $grid;

    $solved = $this->backtrackSolve($grid_copy);

    return $solved ? $grid_copy : FALSE;
  }

  /**
   * Validate shape of the Sudoku grid.
   *
   * @param array $grid
   *   Sudoku grid.
   *
   * @return bool
   *   TRUE if valid shape, FALSE otherwise.
   */
  protected function validShape(array $grid): bool {
    // Must be 9x9.
    if (count($grid) !== 9) {
      return FALSE;
    }

    // Each row must be an array of length 9.
    foreach ($grid as $row) {
      if (!is_array($row) || count($row) !== 9) {
        return FALSE;
      }
    }

    // If we made it here, shape is valid.
    return TRUE;
  }

  /**
   * Find empty cell in the grid. Sets $row and $col to the position if found.
   *
   * @param array $grid
   *   Sudoku grid.
   * @param int $row
   *   Row index.
   * @param int $col
   *   Column index.
   *
   * @return bool
   *   TRUE if found, FALSE if no empty cell.
   */
  protected function findEmpty(array &$grid, int &$row, int &$col): bool {
    // Scan the grid for an empty cell (0).
    for ($r = 0; $r < 9; $r++) {
      for ($c = 0; $c < 9; $c++) {
        if (empty($grid[$r][$c])) {
          $row = $r;
          $col = $c;
          return TRUE;
        }
      }
    }

    // No empty cell found.
    return FALSE;
  }

  /**
   * Check if it's safe to place a particular number at grid[row][col].
   *
   * @param array $grid
   *   Sudoku grid.
   * @param int $row
   *   Row index.
   * @param int $col
   *   Column index.
   * @param int $num
   *   Number to place (1-9).
   *
   * @return bool
   *   TRUE if safe, FALSE otherwise.
   */
  protected function isSafe(
    array $grid,
    int $row,
    int $col,
    int $num,
  ): bool {
    // Check the row.
    for ($c = 0; $c < 9; $c++) {
      if ($grid[$row][$c] == $num) {
        return FALSE;
      }
    }

    // Check the column.
    for ($r = 0; $r < 9; $r++) {
      if ($grid[$r][$col] == $num) {
        return FALSE;
      }
    }

    // Check the 3x3 box region.
    $startRow = $row - $row % 3;
    $startCol = $col - $col % 3;
    for ($r = 0; $r < 3; $r++) {
      for ($c = 0; $c < 3; $c++) {
        if ($grid[$r + $startRow][$c + $startCol] == $num) {
          return FALSE;
        }
      }
    }

    // If we made it here, then it's safe to place this number.
    return TRUE;
  }

  /**
   * Backtracking solver for the Sudoku grid.
   *
   * @param array $grid
   *   Sudoku grid to solve.
   *
   * @return bool
   *   TRUE if solved, FALSE if no solution.
   */
  protected function backtrackSolve(array &$grid): bool {
    $row = $col = 0;
    // If there are no more empty cells, then we've solved the grid.
    if (!$this->findEmpty($grid, $row, $col)) {
      return TRUE;
    }

    // Try numbers 1-9 in the empty cell.
    for ($num = 1; $num <= 9; $num++) {
      if ($this->isSafe($grid, $row, $col, $num)) {
        $grid[$row][$col] = $num;
        if ($this->backtrackSolve($grid)) {
          return TRUE;
        }
        $grid[$row][$col] = 0;
      }
    }

    // Trigger backtrack.
    return FALSE;
  }

  /**
   * Count solutions with a hard cap to detect uniqueness.
   *
   * Returns number of solutions found (0, 1, 2+ depending on cap).
   *
   * @param array $grid
   *   Sudoku grid.
   * @param int $cap
   *   Maximum number of solutions to count (default 2).
   *
   * @return int
   *   Number of solutions found.
   */
  public function countSolutions(array $grid, int $cap = 2): int {
    return $this->countSolutionsRec($grid, $cap);
  }

  /**
   * Recursive helper for counting solutions.
   *
   * @param array $grid
   *   Sudoku grid.
   * @param int $cap
   *   Maximum number of solutions to count.
   * @param int $count
   *   Current count of solutions found.
   *
   * @return int
   *   Number of solutions found.
   */
  protected function countSolutionsRec(
    array &$grid,
    int $cap,
    int &$count = 0,
  ): int {
    // Early exit if we've reached the cap.
    if ($count >= $cap) {
      return $count;
    }
    // Find an empty cell.
    $row = $col = 0;
    if (!$this->findEmpty($grid, $row, $col)) {
      $count++;
      return $count;
    }
    // Try numbers 1-9 in the empty cell.
    for ($num = 1; $num <= 9; $num++) {
      if ($this->isSafe($grid, $row, $col, $num)) {
        $grid[$row][$col] = $num;
        $this->countSolutionsRec($grid, $cap, $count);
        $grid[$row][$col] = 0;
        if ($count >= $cap) {
          return $count;
        }
      }
    }
    // Return the number of solutions found.
    return $count;
  }

}
