<?php

namespace Drupal\supported_image\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\supported_image\Plugin\Field\FieldType\SupportedImageItem;

/**
 * Plugin implementation of the 'supported_image' formatter.
 */
trait SupportedImageFormatterTrait {

  /**
   * Gets the name of the theme hook to be used to render each field item.
   *
   * @return string
   *   The name of the theme hook to be used to render each field item.
   */
  abstract protected function getItemRenderThemeHook(): string;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'display_caption' => TRUE,
      'display_attribution' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);

    $element['display_caption'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display <em>Caption</em> field'),
      '#default_value' => $this->getSetting('display_caption'),
    ];

    $element['display_attribution'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display <em>Attribution</em> field'),
      '#default_value' => $this->getSetting('display_attribution'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();

    $summary[] = $this->getSetting('display_caption')
      ? $this->t('Display <em>Caption</em> field')
      : $this->t('Hide <em>Caption</em> field');

    $summary[] = $this->getSetting('display_attribution')
      ? $this->t('Display <em>Attribution</em> field')
      : $this->t('Hide <em>Attribution</em> field');

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = parent::viewElements($items, $langcode);

    foreach ($items as $delta => $item) {
      if (isset($elements[$delta]['#item']) && $elements[$delta]['#item'] instanceof SupportedImageItem) {
        // Switch to this formatter's render theme hook.
        $elements[$delta]['#theme'] = $this->getItemRenderThemeHook();
        $elements[$delta]['#display_caption'] = $this
          ->getSetting('display_caption');
        $elements[$delta]['#display_attribution'] = $this
          ->getSetting('display_attribution');
      }
    }

    return $elements;
  }

}
