<?php

namespace Drupal\supported_image\Plugin\Field\FieldType;

use Drupal\Core\TypedData\ComputedItemListTrait;
use Drupal\field\FieldConfigInterface;
use Drupal\file\Plugin\Field\FieldType\FileFieldItemList;

/**
 * A list of 'supported_image' field items.
 *
 * Handles the default image functionality in a way that does not require
 * other modules to provide special-handling rules for this field type, as
 * currently happens with the core image field.
 *
 * Adapted from core ImageFormatterBase::getEntitiesToView() and a patch by wim
 * leers from Drupal.org issue 3005528 (comment 4).
 *
 * Users of this field item list can check the first value's is_default_value
 * computed property to determine if they're looking at a real set of values
 * or the field's default (display) value.
 *
 * @see https://www.drupal.org/project/drupal/issues/3435906
 * @see https://git.drupalcode.org/project/drupal/-/commit/3cf0c074d0c75cbc36b63b9bee1f96ae8b079beb#b9e4bc297598422f9c515e20a01f9bcf02867cb3_417_426
 * @see Drupal\image\Plugin\Field\FieldFormatter\ImageFormatterBase::getEntitiesToView()
 * @see https://www.drupal.org/project/drupal/issues/3005528#comment-12808650
 * @see \Drupal\supported_image\Plugin\Field\FieldType\SupportedImageItem::preSave()
 * @see \Drupal\supported_image\Plugin\Field\FieldWidget\SupportedImageWidget::form()
 */
class SupportedImageFieldItemList extends FileFieldItemList {

  use ComputedItemListTrait;

  /**
   * {@inheritdoc}
   */
  protected function computeValue() {
    // Add the default image if needed.
    if (parent::isEmpty()) {
      $default_image = $this
        ->getFieldDefinition()
        ->getSetting('default_image') + SupportedImageItem::defaultFieldSettings()['default_image'];
      // If we are dealing with a configurable field, look in both
      // instance-level and field-level settings.
      if (empty($default_image['uuid']) && $this->getFieldDefinition() instanceof FieldConfigInterface) {
        $default_image = $this
          ->getFieldDefinition()
          ->getFieldStorageDefinition()
          ->getSetting('default_image') + SupportedImageItem::defaultStorageSettings()['default_image'];
      }
      if (!empty($default_image['uuid']) && $file = \Drupal::service('entity.repository')->loadEntityByUuid('file', $default_image['uuid'])) {
        $this->list[0] = $this->createItem(0, [
          'target_id' => $file->id(),
          'alt' => $default_image['alt'],
          'title' => $default_image['title'],
          'caption_value' => $default_image['caption']['value'],
          'caption_format' => $default_image['caption']['format'],
          'attribution_value' => $default_image['attribution']['value'],
          'attribution_format' => $default_image['attribution']['format'],
          'width' => $default_image['width'],
          'height' => $default_image['height'],
          'entity' => $file,
          '_loaded' => TRUE,
          '_is_default' => TRUE,
        ]);
        if ($file->_referringItem) {
          // If the file entity is already being referenced by another field
          // item, clone it so that _referringItem is set to the correct item
          // in each instance.
          $file = clone $file;
          $this->list[0]->entity = $file;
        }
        $file->_referringItem = $this->list[0];
      }
    }
  }

}
