<?php

namespace Drupal\supported_image\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldFilteredMarkup;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\image\Plugin\Field\FieldType\ImageItem;

/**
 * Plugin implementation of the 'supported_image' field type.
 */
#[FieldType(
  id: "supported_image",
  label: new TranslatableMarkup("Supported image"),
  description: [
    new TranslatableMarkup("For uploading images with supporting caption and attribution information"),
    new TranslatableMarkup("Extends the core image field"),
  ],
  category: "file_upload",
  default_widget: "supported_image_image",
  default_formatter: "supported_image",
  list_class: SupportedImageFieldItemList::class,
  constraints: ["ReferenceAccess" => [], "FileValidation" => []],
  column_groups: [
    "file" => [
      "label" => new TranslatableMarkup("File"),
      "columns" => [
        "target_id",
        "width",
        "height",
      ],
      "require_all_groups_for_translation" => TRUE,
    ],
    "alt" => [
      "label" => new TranslatableMarkup("Alt"),
      "translatable" => TRUE,
    ],
    "title" => [
      "label" => new TranslatableMarkup("Title"),
      "translatable" => TRUE,
    ],
    "caption_value" => [
      "label" => new TranslatableMarkup("Caption"),
      "translatable" => TRUE,
    ],
    "caption_format" => [
      "label" => new TranslatableMarkup("Caption text format"),
      "translatable" => FALSE,
    ],
    "attribution_value" => [
      "label" => new TranslatableMarkup("Attribution"),
      "translatable" => TRUE,
    ],
    "attribution_format" => [
      "label" => new TranslatableMarkup("Attribution text format"),
      "translatable" => FALSE,
    ],
  ]
)]
class SupportedImageItem extends ImageItem {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    $settings = parent::defaultStorageSettings();

    $settings['default_image']['caption'] = [
      'value' => '',
      'format' => NULL,
    ];
    $settings['default_image']['attribution'] = [
      'value' => '',
      'format' => NULL,
    ];

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    $settings = [
      'caption_field' => TRUE,
      'caption_field_required' => FALSE,
      'caption_field_allowed_text_formats' => [],
      'caption_field_description' => "A small amount of text that identifies subjects, provides context, or offers additional information about the image. See <a href=\"https://en.wikipedia.org/wiki/Wikipedia:Manual_of_Style/Captions\" target=\"_blank\">Wikipedia's caption style guide</a> for helpful tips and examples.",
      'attribution_field' => TRUE,
      'attribution_field_required' => FALSE,
      'attribution_field_allowed_text_formats' => [],
      'attribution_field_description' => "Credit the author or copyright holder of the image, as may be required by the image's license. See <a href=\"https://wiki.creativecommons.org/wiki/Recommended_practices_for_attribution#Examples_of_attribution\" target=\"_blank\">examples of attribution</a> and consult the image's author/source/license for guidance on what to include here.",
    ] + parent::defaultFieldSettings();

    $settings['default_image']['caption'] = [
      'value' => '',
      'format' => NULL,
    ];
    $settings['default_image']['attribution'] = [
      'value' => '',
      'format' => NULL,
    ];

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = parent::schema($field_definition);

    $schema['columns']['caption_value'] = [
      'description' => 'Supporting caption text content.',
      'type' => 'text',
      'size' => 'big',
    ];
    $schema['columns']['caption_format'] = [
      'description' => 'The text format of the caption value.',
      'type' => 'varchar_ascii',
      'length' => 255,
    ];
    $schema['indexes']['caption_format'] = ['caption_format'];
    $schema['columns']['attribution_value'] = [
      'description' => "Information supplied by the image's licensor.",
      'type' => 'text',
      'size' => 'big',
    ];
    $schema['columns']['attribution_format'] = [
      'description' => 'The text format of the attribution value.',
      'type' => 'varchar_ascii',
      'length' => 255,
    ];
    $schema['indexes']['attribution_format'] = ['attribution_format'];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = parent::propertyDefinitions($field_definition);

    $properties['is_default_value'] = DataDefinition::create('boolean')
      ->setLabel(new TranslatableMarkup('Value is the default value'))
      ->setDescription(new TranslatableMarkup('Flag indicating whether this item is the field default (display) value.'))
      ->setComputed(TRUE)
      ->setClass('\Drupal\supported_image\ItemIsDefaultValue')
      ->setInternal(FALSE);

    $properties['caption_value'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Caption value'))
      ->setDescription(new TranslatableMarkup('Supporting caption text content.'));

    $properties['caption_format'] = DataDefinition::create('filter_format')
      ->setLabel(new TranslatableMarkup('Caption text format'))
      ->setDescription(new TranslatableMarkup('The text format of the caption value.'))
      ->setSetting('allowed_formats', $field_definition->getSetting('caption_field_allowed_text_formats'));

    $properties['caption_processed'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Caption processed'))
      ->setDescription(new TranslatableMarkup('The caption value with its text format applied.'))
      ->setComputed(TRUE)
      ->setClass('\Drupal\supported_image\TextProcessed')
      ->setSetting('text source', 'caption_value')
      ->setSetting('format source', 'caption_format')
      ->setInternal(FALSE);

    $properties['attribution_value'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Attribution value'))
      ->setDescription(new TranslatableMarkup("Information supplied by the image's licensor."));

    $properties['attribution_format'] = DataDefinition::create('filter_format')
      ->setLabel(new TranslatableMarkup('Attribution text format'))
      ->setDescription(new TranslatableMarkup('The text format of the attribution value.'))
      ->setSetting('allowed_formats', $field_definition->getSetting('attribution_field_allowed_text_formats'));

    $properties['attribution_processed'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Attribution processed'))
      ->setDescription(new TranslatableMarkup('The attribution value with its text format applied.'))
      ->setComputed(TRUE)
      ->setClass('\Drupal\supported_image\TextProcessed')
      ->setSetting('text source', 'attribution_value')
      ->setSetting('format source', 'attribution_format')
      ->setInternal(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    // Get base form from ImageItem.
    $element = parent::fieldSettingsForm($form, $form_state);

    $settings = $this->getSettings();

    $element['caption_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable <em>Caption</em> field'),
      '#default_value' => $settings['caption_field'],
      '#description' => $this->t("A small amount of additional information, context, or descriptive content typically shown below the image."),
      '#weight' => 13,
    ];
    $element['caption_field_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('<em>Caption</em> field required'),
      '#default_value' => $settings['caption_field_required'],
      '#states' => [
        'visible' => [
          ':input[name="settings[caption_field]"]' => ['checked' => TRUE],
        ],
      ],
      '#weight' => 14,
    ];
    $element['caption_field_allowed_text_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('<em>Caption</em> field allowed text formats'),
      '#options' => $this->get('caption_format')->getPossibleOptions(),
      '#default_value' => !empty($settings['caption_field_allowed_text_formats']) ? $settings['caption_field_allowed_text_formats'] : [],
      '#description' => $this->t('Select the allowed text formats for the <em>Caption</em> field. If no formats are selected, all available text formats will be displayed to the user.'),
      '#element_validate' => [['\Drupal\text\Plugin\Field\FieldType\TextItemBase', 'validateAllowedFormats']],
      '#states' => [
        'visible' => [
          ':input[name="settings[caption_field]"]' => ['checked' => TRUE],
        ],
      ],
      '#weight' => 15,
    ];

    $element['caption_field_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('<em>Caption</em> field help text'),
      '#default_value' => $settings['caption_field_description'],
      '#rows' => 3,
      '#description' => $this->t('Instructions to present to the user below this field on the editing form.<br />Allowed HTML tags: @tags', ['@tags' => FieldFilteredMarkup::displayAllowedTags()]) . '<br />' . $this->t('This field supports tokens.'),
      '#weight' => 16,
    ];

    $element['attribution_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable <em>Attribution</em> field'),
      '#default_value' => $settings['attribution_field'],
      '#description' => $this->t("Credit the author or copyright holder of the image, as may be required by the image's license."),
      '#weight' => 17,
    ];
    $element['attribution_field_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('<em>Attribution</em> field required'),
      '#default_value' => $settings['attribution_field_required'],
      '#states' => [
        'visible' => [
          ':input[name="settings[attribution_field]"]' => ['checked' => TRUE],
        ],
      ],
      '#weight' => 18,
    ];
    $element['attribution_field_allowed_text_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('<em>Attribution</em> field allowed text formats'),
      '#options' => $this->get('attribution_format')->getPossibleOptions(),
      '#default_value' => !empty($settings['attribution_field_allowed_text_formats']) ? $settings['attribution_field_allowed_text_formats'] : [],
      '#description' => $this->t('Select the allowed text formats for the <em>Attribution</em> field. If no formats are selected, all available text formats will be displayed to the user.'),
      '#element_validate' => [['\Drupal\text\Plugin\Field\FieldType\TextItemBase', 'validateAllowedFormats']],
      '#states' => [
        'visible' => [
          ':input[name="settings[attribution_field]"]' => ['checked' => TRUE],
        ],
      ],
      '#weight' => 19,
    ];

    $element['attribution_field_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('<em>Attribution</em> field help text'),
      '#default_value' => $settings['attribution_field_description'],
      '#rows' => 3,
      '#description' => $this->t('Instructions to present to the user below this field on the editing form.<br />Allowed HTML tags: @tags', ['@tags' => FieldFilteredMarkup::displayAllowedTags()]) . '<br />' . $this->t('This field supports tokens.'),
      '#weight' => 20,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition) {
    $random = new Random();

    $values = [
      'caption_value' => $random->sentences(8),
      'caption_format' => filter_fallback_format(),
      'attribution_value' => $random->sentences(8),
      'attribution_format' => filter_fallback_format(),
    ] + parent::generateSampleValue($field_definition);

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultImageForm(array &$element, array $settings) {
    parent::defaultImageForm($element, $settings);

    $element['default_image']['caption'] = [
      '#type' => 'text_format',
      '#base_type' => 'textarea',
      '#format' => $settings['default_image']['caption']['format'] ?? NULL,
      '#title' => $this->t('Caption'),
      '#description' => isset($settings['caption_field_description']) ? FieldFilteredMarkup::create(\Drupal::token()->replace((string) $settings['caption_field_description'])) : NULL,
      '#default_value' => $settings['default_image']['caption']['value'] ?? NULL,
      '#rows' => 2,
      '#attributes' => ['class' => ['js-text-full', 'text-full']],
    ];

    $element['default_image']['attribution'] = [
      '#type' => 'text_format',
      '#base_type' => 'textarea',
      '#format' => $settings['default_image']['attribution']['format'] ?? NULL,
      '#title' => $this->t('Attribution'),
      '#description' => isset($settings['attribution_field_description']) ? FieldFilteredMarkup::create(\Drupal::token()->replace((string) $settings['attribution_field_description'])) : NULL,
      '#default_value' => $settings['default_image']['attribution']['value'] ?? NULL,
      '#rows' => 2,
      '#attributes' => ['class' => ['js-text-full', 'text-full']],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    parent::preSave();
    // Prevent the field default (display) value from getting saved as actual
    // field data.
    if ($this->get('is_default_value')->getValue()) {
      $this->setValue(NULL);
    }
  }

}
