<?php

namespace Drupal\supported_image\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldFilteredMarkup;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\image\Plugin\Field\FieldWidget\ImageWidget;

/**
 * Plugin implementation of the 'supported_image_image' widget.
 */
#[FieldWidget(
  id: 'supported_image_image',
  label: new TranslatableMarkup('Supported image'),
  description: new TranslatableMarkup('Image with supporting caption and attribution information.'),
  field_types: ['supported_image'],
)]
class SupportedImageWidget extends ImageWidget {

  /**
   * {@inheritdoc}
   */
  public function form(FieldItemListInterface $items, array &$form, FormStateInterface $form_state, $get_delta = NULL) {
    // Prevent the field default (display) value from coming through in the
    // widget form.
    if ($items->first()?->get('is_default_value')->getValue()) {
      $items->setValue(NULL);
    }

    return parent::form($items, $form, $form_state, $get_delta);
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Add properties needed by the process() method.
    $field_settings = $this->getFieldSettings();
    $element['#caption_field'] = $field_settings['caption_field'];
    $element['#caption_field_required'] = $field_settings['caption_field_required'];
    $element['#caption_field_allowed_text_formats'] = $field_settings['caption_field_allowed_text_formats'];
    $element['#caption_field_description'] = $field_settings['caption_field_description'];
    $element['#attribution_field'] = $field_settings['attribution_field'];
    $element['#attribution_field_required'] = $field_settings['attribution_field_required'];
    $element['#attribution_field_allowed_text_formats'] = $field_settings['attribution_field_allowed_text_formats'];
    $element['#attribution_field_description'] = $field_settings['attribution_field_description'];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function process($element, FormStateInterface $form_state, $form) {
    $element = parent::process($element, $form_state, $form);

    $item = $element['#value'];
    $item['fids'] = $element['fids']['#value'];

    // Initially, $element['#value'] will be in the expected, "massaged",
    // field item value format. But, in AJAX operations, $element['#value']
    // will be in the raw, "un-massaged" format and will need expansion.
    $item['caption_format'] = $item['caption_format'] ?? $item['caption']['format'] ?? NULL;
    $item['caption_value'] = $item['caption_value'] ?? $item['caption']['value'] ?? NULL;
    $item['attribution_format'] = $item['attribution_format'] ?? $item['attribution']['format'] ?? NULL;
    $item['attribution_value'] = $item['attribution_value'] ?? $item['attribution']['value'] ?? NULL;

    // Add the additional caption field.
    $element['caption'] = [
      '#type' => 'text_format',
      '#base_type' => 'textarea',
      '#format' => $item['caption_format'],
      '#allowed_formats' => !empty($element['#caption_field_allowed_text_formats']) ? $element['#caption_field_allowed_text_formats'] : NULL,
      '#title' => new TranslatableMarkup('Caption'),
      '#description' => isset($element['#caption_field_description']) ? FieldFilteredMarkup::create(\Drupal::token()->replace((string) $element['#caption_field_description'])) : NULL,
      '#default_value' => $item['caption_value'],
      '#rows' => 2,
      '#attributes' => ['class' => ['js-text-full', 'text-full']],
      '#access' => (bool) $item['fids'] && $element['#caption_field'],
      '#required' => $element['#caption_field_required'],
      '#element_validate' => $element['#caption_field_required'] ? [[static::class, 'validateRequiredFields']] : [],
      '#weight' => -10,
    ];

    // Add the additional attribution field.
    $element['attribution'] = [
      '#type' => 'text_format',
      '#base_type' => 'textarea',
      '#format' => $item['attribution_format'],
      '#allowed_formats' => !empty($element['#attribution_field_allowed_text_formats']) ? $element['#attribution_field_allowed_text_formats'] : NULL,
      '#title' => new TranslatableMarkup('Attribution'),
      '#description' => isset($element['#attribution_field_description']) ? FieldFilteredMarkup::create(\Drupal::token()->replace((string) $element['#attribution_field_description'])) : NULL,
      '#default_value' => $item['attribution_value'],
      '#rows' => 2,
      '#attributes' => ['class' => ['js-text-full', 'text-full']],
      '#access' => (bool) $item['fids'] && $element['#attribution_field'],
      '#required' => $element['#attribution_field_required'],
      '#element_validate' => $element['#attribution_field_required'] ? [[static::class, 'validateRequiredFields']] : [],
      '#weight' => -9,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $values = parent::massageFormValues($values, $form, $form_state);
    foreach ($values as &$value) {
      if (isset($value['caption'])) {
        // Expand caption into caption_value and caption_format values.
        $value['caption_value'] = $value['caption']['value'] ?? NULL;
        $value['caption_format'] = $value['caption']['format'] ?? NULL;
        unset($value['caption']);
      }
      if (isset($value['attribution'])) {
        // Expand attribution into attribution_value and attribution_format
        // values.
        $value['attribution_value'] = $value['attribution']['value'] ?? NULL;
        $value['attribution_format'] = $value['attribution']['format'] ?? NULL;
        unset($value['attribution']);
      }
    }
    return $values;
  }

}
