<?php

namespace Drupal\supported_image;

use Drupal\Core\TypedData\DataDefinitionInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Render\FilteredMarkup;
use Drupal\text\TextProcessed as CoreTextProcessed;

/**
 * A computed property for processing text with a format.
 *
 * Required settings (below the definition's 'settings' key) are:
 *  - text source: The property containing the text to be processed.
 *  - format source: The property containing the text format.
 *
 * Adapted from Drupal\text\TextProcessed to include a configurable format
 * source.
 *
 * @see Drupal\supported_image\Plugin\Field\FieldType\SupportedImageItem::propertyDefinitions()
 */
class TextProcessed extends CoreTextProcessed {

  /**
   * {@inheritdoc}
   */
  public function __construct(DataDefinitionInterface $definition, $name = NULL, ?TypedDataInterface $parent = NULL) {
    parent::__construct($definition, $name, $parent);

    if ($definition->getSetting('format source') === NULL) {
      throw new \InvalidArgumentException("The definition's 'format source' key has to specify the name of the property holding the text format.");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    if ($this->processed !== NULL) {
      return FilteredMarkup::create($this->processed->getProcessedText());
    }

    $item = $this->getParent();
    $text = $item->{($this->definition->getSetting('text source'))};
    $format = $item->{($this->definition->getSetting('format source'))};

    // Avoid doing unnecessary work on empty strings.
    if (!isset($text) || $text === '') {
      $this->processed = new FilterProcessResult('');
    }
    else {
      $build = [
        '#type' => 'processed_text',
        '#text' => $text,
        '#format' => $format,
        '#filter_types_to_skip' => [],
        '#langcode' => $item->getLangcode(),
      ];
      // Capture the cacheability metadata associated with the processed text.
      $processed_text = $this->getRenderer()->renderInIsolation($build);
      $this->processed = FilterProcessResult::createFromRenderArray($build)->setProcessedText((string) $processed_text);
    }
    return FilteredMarkup::create($this->processed->getProcessedText());
  }

}
